#include "SctData/TrimRangeTestResult.h"
#include "sctConf/configipc.h"
#include "../../AnalysisTestFramework.h"
#include "../../TestFileUtil.h"
#include "../../CutUtils.h"

using namespace SctConfiguration;

struct ChipInfo {
    float charge;
    float SCTDAQVersion;
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int chip;
    unsigned int nScans;
    short type;
    unsigned short algorithm;
    char serial[15];    
};
string ChipInfoStr = "Charge/F:SCTDAQVersion:Run/i:Scan:Chip:NScans:Type/S:Alg/s:Serial/C";

struct ChannelInfo {
    float charge;    
    float SCTDAQVersion;
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int channel;
    unsigned int nScans;
    short type;
    unsigned short algorithm;    
    char serial[15];    
};
string ChannelInfoStr = "Charge/F:SCTDAQVersion:Run/i:Scan:Channel:NScans:Type/S:Alg/s:Serial/C";


struct TRChipData {
    float target;
    float meanVt50;
    float rmsVt50;
    float meanOffset;
    float rmsOffset;
    float meanStep;
    float rmsStep;
    int range;
    int ntrim;
    int pass;
};
string ChipDataStr = "Target/F:MeanVt:RmsVt:MeanOffset:RmsOffset:MeanStep:RmsStep:Range/I:NTrim:Pass";

struct TRChannelData {    
    int trim;
    int status;
    float vthr;
    float offset;
    float step;
};
string ChannelDataStr = "Trim/I:Status:Vt/F:Offset:Step";


//Globals variables:
TRChipData RodDaq;                    //Our data
TRChipData SctDaq;                    //SCTDAQ data
ChipInfo info;                          //Some info

TRChannelData chRodDaq;	
TRChannelData chSctDaq;	
ChannelInfo chInfo;

//Noise Occupancy stuff
string Ext = "_Trim.dat";
string Output = "${SCT_DAQ_ROOT}/SystemTests/logs/TrimRange.root";
string Arg = "-6";

class TrimRangeCompare : public AnalysisTestFramework<TrimRangeTestResult> {
public:
    TrimRangeCompare();
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const TrimRangeTestResult& t);   
    virtual float compareChip(const TrimRangeTestResult& t);   	
    virtual void compareChannel(const TrimRangeTestResult& t, float version);   
    virtual void summaryOutput();
    virtual void setup();
    
    ConfigIPC config;
    TTree* channelTree;
    
private:
    void checkShapeForEachRange(TTree& t, double prob, string name, string var, string cut="1");
};


string getFileName(string serial) {
    ostringstream oss;
    oss << serial << Ext;
    
    return oss.str();
}

void TrimRangeCompare::summaryOutput() {
    //Value checks
    if (cut(*tree, "pass", "(RodDaq.Pass-SctDaq.Pass)", 0.01, 0.01, 0.01, true, "Info.SCTDAQVersion>3.425") > 0) {
        ++exitCode;
        cout << "Failed pass tail check" << endl;
    }
    //Only check for versions>3.42
    if (cut(*tree, "ntrim", "(RodDaq.NTrim-SctDaq.NTrim)", 0.01, 0.01, 0.01, true, "Info.SCTDAQVersion>3.425") > 0) {
        ++exitCode;
        cout << "Failed ntrim tail check" << endl;
    } 
    //Awaits a new version of SCTDAQ
    if (cut(*tree, "range", "(RodDaq.Range-SctDaq.Range)", 0.1, 0.1, 0.3, true, "Info.SCTDAQVersion>3.425") > 0.1) {
        ++exitCode;
        cout << "Failed range tail check" << endl;
    } 
    if (cut(*tree, "target", "(RodDaq.Target-SctDaq.Target)", 0.01, 0.01, 0.01, true, "Info.SCTDAQVersion>3.425") > 0) {
        ++exitCode;
        cout << "Failed target tail check" << endl;
    } 
    if (cut(*tree, "meanVt50", "(RodDaq.MeanVt-SctDaq.MeanVt)", 1, 1, 0.01, true, "Info.SCTDAQVersion>3.425") > 0) {
        ++exitCode;
        cout << "Failed meanVt50 tail check" << endl;
    } 
    
    //Distribution shape checks
    double val = 0;
    val = compareShape(*tree, "rangeShape", "RodDaq.Range", "SctDaq.Range", "Info.SCTDAQVersion>3.425");
    //cout << "Range shape: " << (val*100) << "%" << endl;
    if (val > 0 && val < 0.9) {
	++exitCode;
	cout << "Failed range shape check: " << val << endl;
    }
    
    val = compareShape(*channelTree, "trimValShape", "RodDaq.Trim", "SctDaq.Trim", "Info.SCTDAQVersion>3.425");
    //cout << "TrimVal shape: " << (val*100) << "%" << endl;
    if (val > 0 && val < 0.9) {
	++exitCode;
	cout << "Failed TrimVal shape check: " << val << endl;
    }
    
    checkShapeForEachRange(*tree, 0.9, "StepMeanShape", "MeanStep", "Info.SCTDAQVersion>3.425");
    checkShapeForEachRange(*tree, 0.9, "StepRmsShape", "RmsStep", "Info.SCTDAQVersion>3.425");
    checkShapeForEachRange(*tree, 0.9, "OffsetMeanShape", "MeanOffset", "Info.SCTDAQVersion>3.425");
    checkShapeForEachRange(*tree, 0.9, "OffsetRmsShape", "RmsOffset", "Info.SCTDAQVersion>3.425");
    checkShapeForEachRange(*tree, 0.9, "Vt50RmsShape", "RmsVt", "Info.SCTDAQVersion>3.425");
    
    exitCode += errorCode;
}

void TrimRangeCompare::checkShapeForEachRange(TTree& t, double prob, string name, string var, string cut) {    
    string rodVar = "RodDaq." + var;
    string sctVar = "SctDaq." + var;
    
    for (unsigned int i=0; i<4; ++i) {
	ostringstream ossName;
	ossName << name << i;
	string curName = ossName.str();
	ostringstream oss;
	oss << "RodDaq.Range==" << i << " && SctDaq.Range==" << i << " && " << cut;
	string curCut = oss.str();
	double val = compareShape(t, curName, rodVar, sctVar, curCut);
	if (val > 0 && val < prob) {
	    ++exitCode;
	    cout << "Failed " << name << " shape check for range " << i << " Prob: " << val << endl;
	}
    }
}

void TrimRangeCompare::compare(const TrimRangeTestResult& sd) {
    float version = compareChip(sd);
    compareChannel(sd, version);
}

float TrimRangeCompare::compareChip(const TrimRangeTestResult& sd) {
    strncpy(info.serial, sd.getModuleName().c_str(), 14);
    info.runNumber = sd.getRunNumber();
    info.scanNumber = sd.getScanNumberAt(0);
    info.nScans = sd.getNScans();
    info.charge = sd.charge;
    info.type = sd.type;
    info.algorithm = sd.algorithm;
    
    string fileName = getFileName(sd.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    
    SctDaq.pass = TestFileUtil::getPass(file);
    info.SCTDAQVersion = TestFileUtil::skipHeader(file);

    for (unsigned int chip = 0; chip<nChipModule; ++chip) {
	info.chip = chip;
	string chipStr;
	file >> chipStr >> SctDaq.range >> SctDaq.target >> SctDaq.ntrim >> SctDaq.meanVt50 >> SctDaq.rmsVt50 
	     >> SctDaq.meanStep >> SctDaq.rmsStep >> SctDaq.meanOffset >> SctDaq.rmsOffset;
	//Ignore rest of line
	file.ignore(256, '\n');
	
	RodDaq.pass = sd.getPassed()?(sd.getProblem()?2:1):0;
	RodDaq.range = sd.chipTrim[chip]->range;
	RodDaq.target = sd.chipTrim[chip]->target;
	RodDaq.meanVt50 = sd.chipTrim[chip]->channelTrim.mean().vthr;
	RodDaq.rmsVt50 = sqrt(sd.chipTrim[chip]->channelTrim.var().vthr);
	RodDaq.meanOffset = sd.chipTrimData[chip]->getOffsets().mean();
	RodDaq.rmsOffset = sqrt(sd.chipTrimData[chip]->getOffsets().var());
	RodDaq.meanStep = sd.chipTrimData[chip]->getSteps().mean();
	RodDaq.rmsStep = sqrt(sd.chipTrimData[chip]->getSteps().var());
	RodDaq.ntrim = sd.chipTrimData[chip]->channelData.n();

        tree->Fill();
    }
    return info.SCTDAQVersion;
}


void TrimRangeCompare::compareChannel(const TrimRangeTestResult& sd, float version) {
    ModuleConfiguration mc;
    mc.getABCDModule() = config.getModuleConfig(sd.getModuleName());
    
    strncpy(chInfo.serial, sd.getModuleName().c_str(), 14);
    chInfo.runNumber = sd.getRunNumber();
    chInfo.scanNumber = sd.getScanNumberAt(0);
    chInfo.nScans = sd.getNScans();
    chInfo.charge = sd.charge;
    chInfo.type = sd.type;
    chInfo.algorithm = sd.algorithm;
    chInfo.SCTDAQVersion = version;
    
    vector<int> validChannels(nChannelModule, 1);
    
    string fileName = sd.getModuleName() + ".mask";
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open mask file: " + fileName, __FILE__, __LINE__);
    
    while (!file.eof()) {
	unsigned int channel = 0;
	file >> channel;
	if (!file.good()) continue;
	//Ignore rest of line
	file.ignore(256, '\n');
	
	validChannels[channel] = 0;
    }
    
    
    for (unsigned int chip=0; chip<nChipModule; ++chip) {
	for (unsigned int channel=0; channel<nChannelChip; ++channel) {
	    chInfo.channel = chip*nChannelChip + channel;
	    chSctDaq.trim = mc.getChipConfiguration(chip).getTrim(channel);
	    chSctDaq.status = validChannels[chip*nChannelChip+channel];
	    
	    chRodDaq.trim = sd.chipTrim[chip]->channelTrim.getAt(channel).value.trim;
	    chRodDaq.status = sd.chipTrim[chip]->channelTrim.getAt(channel).valid;
	    chRodDaq.vthr = sd.chipTrim[chip]->channelTrim.getAt(channel).value.vthr;
	    chRodDaq.offset = sd.chipTrimData[chip]->channelData.getAt(channel).value.p0;
	    chRodDaq.step = 1.f/sd.chipTrimData[chip]->channelData.getAt(channel).value.p1;
	    
	    channelTree->Fill();
	}
    }
}


/**
  Create the TFile and TTree
  */
void TrimRangeCompare::setup() {
    string name = Env::substituteVariables(Output);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("TRChip", "TrimRange Chip Comparison Data");
    channelTree = new TTree("TRChannel", "TrimRange Channel Comparison Data");
    tree->Branch("RodDaq", &RodDaq, ChipDataStr.c_str());
    tree->Branch("SctDaq", &SctDaq, ChipDataStr.c_str());
    tree->Branch("Info", &info, ChipInfoStr.c_str());
    
    channelTree->Branch("RodDaq", &chRodDaq, ChannelDataStr.c_str());
    channelTree->Branch("SctDaq", &chSctDaq, ChannelDataStr.c_str());
    channelTree->Branch("Info", &chInfo, ChannelInfoStr.c_str());    
    
    info.serial[14] = '\0';
    chInfo.serial[14] = '\0';
}

/**
  Downloads the response curve data
  */
void TrimRangeCompare::downloadData(string serialNumber) {
    ostringstream oss;
    ostringstream oss1;
    TestInfo info = moduleData.getTrimInfo(serialNumber);
    oss << "java ProdDatabase/getDBfiles " << Arg << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    oss1 << "java ProdDatabase/getDBfiles -trim " << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    system(oss.str().c_str());
    system(oss1.str().c_str());
}

/**
  Publishes all the data we need, looping over modules,
  then waits for it all to be fitted, analyzed
  */
void TrimRangeCompare::publishData(SctTestApiStatus* status) {
    highLevelApi->trim(status);
}

TrimRangeCompare::TrimRangeCompare() {
    channelTree = 0;
}


int main(int argc, char** argv) {
    TrimRangeCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}

