#include "SctData/PipelineTestResult.h"
#include "../../AnalysisTestFramework.h"
#include "../../CutUtils.h"
#include "../../TestFileUtil.h"

struct Info {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int chip;
    unsigned int nScans;
    char serial[15];    
};
string InfoStr = "Run/i:Scan:Chip:NScans:Serial/C";

struct SDData {
    int ngood;
    int pass;
};
string DataStr = "NGood/I:Pass/I";

//Globals variables:
SDData RodDaq;                    //Our data
SDData SctDaq;                    //SCTDAQ data
Info info;                          //Some info

//Noise Occupancy stuff
string Ext = "_Pipeline.dat";
string Output = "${SCT_DAQ_ROOT}/SystemTests/logs/Pipeline.root";
string Arg = "-3";

class PipelineCompare : public AnalysisTestFramework<PipelineTestResult> {
public:
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const PipelineTestResult& t);   	
    virtual void setup();
    virtual void summaryOutput();
};


string getFileName(string serial) {
    ostringstream oss;
    oss << serial << Ext;
    
    return oss.str();
}

void PipelineCompare::compare(const PipelineTestResult& sd) {
    strncpy(info.serial, sd.getModuleName().c_str(), 14);
    info.runNumber = sd.getRunNumber();
    info.scanNumber = sd.getScanNumberAt(0);
    info.nScans = sd.getNScans();
    
    string fileName = getFileName(sd.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    
    SctDaq.pass = TestFileUtil::getPass(file);
    TestFileUtil::skipHeader(file);

    for (unsigned int chip = 0; chip<nChipModule; ++chip) {
	string chipStr;
	info.chip = chip;
	file >> chipStr >> SctDaq.ngood;
	//Ignore rest of line
	file.ignore(256, '\n');

	RodDaq.ngood = nChannelChip - sd.getDefects().getDefectsAffectingElement(ModuleElement::Chip(chip))->getAllDefects().size();
	RodDaq.pass = sd.getPassed();

        tree->Fill();
    }
}


void PipelineCompare::summaryOutput() {
    if (cut(*tree, "ngood", "(RodDaq.NGood-SctDaq.NGood)", 0.01, 0.01, 0.01) > 0) {
	++exitCode;
	cout << "Failed ngood tail check" << endl;
    }   
    if (cut(*tree, "pass", "(RodDaq.Pass-SctDaq.Pass)", 0.01, 0.01, 0.01) > 0) {
	++exitCode;
	cout << "Failed pass tail check" << endl;
    }
    exitCode += errorCode;
}

/**
  Create the TFile and TTree
  */
void PipelineCompare::setup() {
    string name = Env::substituteVariables(Output);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("PipeData", "Pipeline Comparison Data");
    tree->Branch("RodDaq", &RodDaq, DataStr.c_str());
    tree->Branch("SctDaq", &SctDaq, DataStr.c_str());
    tree->Branch("Info", &info, InfoStr.c_str());
    info.serial[14] = '\0';
}

/**
  Downloads the response curve data
  */
void PipelineCompare::downloadData(string serialNumber) {
    ostringstream oss;
    TestInfo info = moduleData.getPipelineInfo(serialNumber);
    oss << "java ProdDatabase/getDBfiles " << Arg << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    system(oss.str().c_str());
}

/**
  Publishes all the data we need, looping over modules,
  then waits for it all to be fitted, analyzed
  */
void PipelineCompare::publishData(SctTestApiStatus* status) {
    highLevelApi->pipeline(status);
}


int main(int argc, char** argv) {
    PipelineCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}












