// File: TApi.h

#ifndef SCTAPI_TAPI_H
#define SCTAPI_TAPI_H

#include <list>
#include <string>
#include <TArrayD.h>
#include <TArrayI.h>
#include <TArrayL.h>
#include <TArrayS.h>

// Declare classes used + define RodLabel
#include "SctApiFwd.h"

#ifndef __CINT__
#include "TApiWrappers.h"
#endif

#include "processor.h"

#include <TObject.h>

// using namespace SctPixelRod;

class TriggerWrapper;
class ScanDefWrapper;

/**
   Wrap Trigger for ROOT
*/
class TTrigger : private TriggerWrapper {
  TTrigger &operator=(const TTrigger &);

 public:
  /// Create default L1A trigger
  TTrigger();

  TTrigger(const TTrigger &);

  /// Call TObject destructor
  virtual ~TTrigger();

  void singleL1A();
  void doubleL1A(int delay);
  void delayedL1A(int delay);
  void calL1A(int delay);
  void pulseL1A(int delay);
  void softL1A(int delay);
  void softCalL1A(int delay, int delay2);
  void softPulseL1A(int delay, int delay2);
  void bcL1A(int delay);
  void bcCalL1A(int delay, int delay2);
  void bcPulseL1A(int delay, int delay2);

  /// Which command value to increment
  int incCmd;

  /// Increment trigger data by (if 0 then don't!)
  int incData;

 private:
  void update();

  friend class TApi;
  friend class TScanDef;

  // For CINT and ROOT
  ClassDef(TTrigger,1)
};

/**
    TScanDef class

    Wrap Scan class for ROOT
*/ 
class TScanDef : private ScanDefWrapper {
  TScanDef &operator=(const TScanDef &);

 public:
  /// Constructor
  TScanDef();

  TScanDef(const TScanDef &);

  /// Call appropriate destructor
  virtual ~TScanDef();

  /// Print the scan (not const as it has to do an update
  void print();

  /// Set up scanPoints
  void configure(UINT16 type, FLOAT32 start, FLOAT32 stop, FLOAT32 step);

  /// Set up scanPoints for set 2
  void configure2(UINT16 type, FLOAT32 start, FLOAT32 stop, FLOAT32 step);

  /// Set individual scanPoints for set 1
  void setScanPoint(int index, FLOAT32 value);

  /// Set individual scanPoints for set 2
  void setScanPoint2(int index, FLOAT32 value);

  /// Set number of triggers for a scan point
  void setTriggersPoint(int index, UINT32 ntrigs);

  /// How many triggers to be sent for each scan point
  long trigsPerBurst;

  /// Which variable to scan over
  int scanVariable;

  /// Which variable to scan second set over
  int scanVariable2;

  /// Sequence of triggers to send to module set 1
  TTrigger trigSequence;

  /// Sequence of triggers to send to module set 2
  TTrigger trigSequence2;

  /// Condensed / expanded
  int full;

  /// 16 / 32
  int bits32;

  /// Loop over the calibration line during the scan
  int loopCalLine;

  /// Which slave distribution to use, see SctApi docs
  int distSlave;

  /// Debug mode (don't kill tasks and don't write data)
  int debug;

  /// Use TIM for triggers
  int tim;

 private:
  /// Synchronise worker with other contents of object
  void update();

  friend class TApi;

  // For CINT and ROOT
  ClassDef(TScanDef,1)
};

/** The Root interface to SCTAPI.

    This class is processed to generate a Root dictionary
    that means it can be used from Root. 
    This is basically a proxy for the SctApi class, all
    methods on it are delegated to an instance of the SctApi
    class which is created on construction.

    A secondary purpose of this class is to extract the public 
    interface from the SctApi class. This means that this class
    should be capable of doing anything it is necessary to do
    using the SctApi class, and also serves the purpose of testing
    the abstraction from the underlying RodModule classes and 
    exceptions.
 */
class TApi : public TObject {
  public:
    /// Constructor
    TApi();

    /// Destructor
    ~TApi();

    /// Initialise everything from configuration 
    /**
        Initialise all the Rods and modules found in the configuration
    */
    void initialiseAll(int runNumber);

    /// Shutdown everything
    void shutdownAll();

    /// Change the run number
    void setRunNumber(int newRun);

    /// Change the scan number
    void setScanNumber(int newScan);

    /*************** ROD Diagnostics ********************/

    /// Echo data to a rod
    /**
       Using a primitive, send some data to the ROD.
       Expect an output.
     */
    void echo(unsigned int partition, unsigned int crate, unsigned int rod,
              unsigned int length, unsigned long *data);

    /// Echo data to all RODs
    /**
       Using a primitive, send some data to the ROD.
       Expect an output.
     */
    void echoAll(unsigned int length, unsigned long *data);

    /// Echo data to a rod
    /**
       Using a primitive, send some data to the ROD.
       Expect an output.
     */
    void echoSlave(unsigned int partition, unsigned int crate, unsigned int rod,
                   unsigned int slave, unsigned int length, unsigned long *data);

    /// Reload configuration from the "configuration database"
    void loadConfiguration();

    /// Load module configurations from the "configuration database"
    void loadModuleConfigurations();

    /// Reset BOC configuration
    void configureBOC(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Wait for a ROD to finish a primlist
    void awaitResponse(unsigned int partition, unsigned int crate, unsigned int rod, int timeout = 5);

    /// Get a response from the rod
    unsigned long *getResponse(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Get a response from the rod
    unsigned long *getResponse(unsigned int partition, unsigned int crate, unsigned int rod, 
                               unsigned long *length);

    /// Get a message in the text buffer
    /**
       Get a message from a ROD text buffer
       Returns whether the action was completed
    */
    bool getRodMessage(unsigned int partition, unsigned int crate, unsigned int rod,
                       char *buffer, unsigned long &length);

    /// Flash the LED associated with a slave DSP. 
    /**
       This creates its own primitive list and sends it to 
       a slave DSP on the ROD.

       Period given in ms
    */
    void flashLED(unsigned int partition, unsigned int crate, unsigned int rod,
                  long slaveNumber, long period = 1000, long flashes = 10);


    void status();

    /***************** Direct ROD access (Debug only) ********************/
    /** Should be able to do anything with these commands */

    /// Read data from a ROD DSP
    /**
       Dump contents of a block of DSP memory to cout.
    */
    Int_t dspBlockDump(unsigned int partition, unsigned int crate, unsigned int rod,
                       long dspStart, long numWords, long dspNumber, bool usePrimitive = true);

    /// Dump data from a ROD DSP to a file
    /**
       Dump contents of a block of DSP memory to a file.
    */
    Int_t dspBlockDumpFile(unsigned int partition, unsigned int crate, unsigned int rod,
                           long dspStart, long numWords, long dspNumber, const char *filename, bool usePrimitive = true);

    /// Read data from a ROD DSP
    /**
       Read data
    */
    unsigned long *dspBlockRead(unsigned int partition, unsigned int crate, unsigned int rod,
                                long dspStart, long numWords, long dspNumber, 
                                unsigned long *length, bool usePrimitive = true);

    /// Write a block of memory to a ROD DSP
    int dspBlockWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                      unsigned long *buffer, unsigned long dspAddress, long numWords, 
                      long dspNumber, bool usePrimitive = true);

    /// Read one ROD Status Register via HPI
    unsigned long readRodStatusReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                   long regNumber);

    /// Read one ROD Command Register via HPI
    unsigned long readRodCommandReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                    long regNumber);

    /// Read a single 32b word from MasterDSP SDRAM via HPI
    unsigned long dspSingleRead(unsigned int partition, unsigned int crate, unsigned int rod,
                                const unsigned long dspAddr, long dspNumber);

    /// Write a single 32b word to MasterDSP SDRAM via HPI
    void dspSingleWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                        unsigned long dspAddr, unsigned long value, long dspNumber);

    /// Create a prim list
    void createDebugPrimList();

    /// Insert a primitive in a prim list
    void addDebugPrimList(unsigned long length, long index, long id, long version,
                          unsigned long * body);

    /// Send the previously created prim list
    void sendDebugPrimList(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Send the previously created prim list to all RODs
    void sendDebugPrimListAll();

    /// Send prim list to slave
    void sendDebugSlavePrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                                unsigned int slave, bool await, bool response);

    void debugPrimListFromFile(const char *fileName);

    void dumpDebugPrimList();

    std::list<SctApi::RodLabel> listRods();

    /// Return the module configuration data for the required module
    unsigned long *retrieveModule(UINT32 mid);

    /**** Official SCTAPI bit (majority unimplemented) ******/

    /// Return unique identifier for a module
    /**
       @param sn Module serial number
       @return The unique identifier, BAD_MODULE if not found
    */
    UINT32 findModule(const char *sn);

    /// Return unique identifier for mur/module 
    UINT32 findModule(INT32 mur, INT32 module);

    /// Return unique identifer from barrel geometry
    UINT32 findBarrelModule(INT32 barrel, INT32 row, INT32 number);

    /// Return unique identifer from endcap geometry
    UINT32 findEndcapModule(INT32 disk, INT32 ring, INT32 number);

    /// Get all modules from ROD to memory
    void getABCDModules(UINT32 bank);

    /// Get module from ROD to memory
    void getABCDModule(UINT32 mid, UINT32 bank);

    /// Set all module configs in a ROD bank
    void setABCDModules(UINT32 bank);

    /// Set module config in ROD bank
    void setABCDModule(UINT32 mid, UINT32 bank);

    /// Send configuration for one module from the ROD bank to the module
    /** 
        @param mid Unique identifier of the module to send
        @param bank Configuration bank in the ROD
        @param type How much configuration to transmit
    */
    void sendABCDModule(UINT32 mid, UINT32 bank, UINT32 type=2);

    /// Send configuration for all modules from the ROD bank
    /** 
        @param bank Configuration bank in the ROD
        @param type How much configuration to transmit
    */
    void sendABCDModules(UINT32 bank, UINT32 type=2);

    /// Modify mask of module in online configuration
    void modifyABCDMask(UINT32 mid, UINT32* mask);

    /// Modify trims of module in online configuration
    void modifyABCDTrims(UINT32 mid, UINT8* trims);

    /// Modify variable of all modules in online configuration
    void modifyABCDVar(UINT32 typ, FLOAT32 var);

    /// Modify variable of module in online configuration
    void modifyABCDVar(UINT32 mid, UINT32 typ, FLOAT32 var);

    /// Modify variable of one chip of module in online configuration
    void modifyABCDVar(UINT32 mid, UINT32 c, UINT32 typ, FLOAT32 var);

    /// Modify variable for specific module/chip in ROD bank
    void modifyABCDVarROD(UINT32 mid, UINT32 chip, UINT32 typ, FLOAT32 var, UINT32 bank);

    /// Modify variable in ROD bank
    void modifyABCDVarROD(UINT32 typ, FLOAT32 var, UINT32 bank);

    /// Modify a BOC variable
    void modifyBOCParam(unsigned int partition, unsigned int crate, unsigned int rod,
                        unsigned int channel, unsigned int type, unsigned int val);

    /// Do a scan (hardcoded for testing)
    void defaultScan(int type);

    /// Tidy up just in case things got left behind
    void tidyHistogramming();

    /// Carry out the scan defined by scan
    void doScan(TScanDef scan);

    void awaitScan();

    /// Scan returning histograms of raw data
    void doRawScan(TScanDef scan, int delay, bool configure = true, bool clkBy2 = false);

    /// Send one trigger to the currently configured modules
    void sendTrigger(unsigned int partition, unsigned int crate, unsigned int rod, TTrigger trig);

    /// Print the module configuration in memory
    /** 
       Uses the sctConf method
    */
    void printABCDModule(int mid);

    /// Print the module configuration from the ROD 
    /**
       Uses the sctConf method
    */
    void printABCDRodModule(int mid, int bank);

    /// Print BOC setup of all channels
    void printBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

    void currentBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Print all BOC registers 
    void printBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

    void currentBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Save BOC setup
    void saveBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Save BOC registers
    void saveBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Response BOC setup
    void restoreBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Restore BOC registers
    void restoreBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Print raw link data
    void rawData(unsigned int partition, unsigned int crate, unsigned int rod, int delay, int units, bool setMask = true, TTrigger trig = TTrigger());

    /// Probe channels on a ROD to find out what's there
    char *probe(unsigned int partition, unsigned int crate, unsigned int rod, signed int harness = -1);

    /// Probe channels on a ROD to find out what's there
    char *probeWithTrigger(unsigned int partition, unsigned int crate, unsigned int rod, TTrigger trig, signed int harness = -1);

    /// Probe channels on a ROD to find out what's there
    void probeScan(unsigned int partition, unsigned int crate, unsigned int rod, 
                   TScanDef scan, signed int harness = -1);

    /// Probe all modules and check if value returned is in string
    bool checkAllModulesProbe(const char *value);

    /// Attempt to configure
    void autoConfigure();

    /// Print "BOC Histogram"
    void bocHistogram(unsigned int partition, unsigned int crate, unsigned int rod, 
                      unsigned int samples, unsigned int numLoops);

    /// Do something to the rod mode!
    void rodMode(unsigned int partition, unsigned int crate, unsigned int rod,
                 int mode, int flag, int fifoSetup, int nBins, int delay, int message);

    void standardRegisterDump(unsigned int partition, unsigned int crate, unsigned int rod);

    /// Turn all the laser currents off 
    /**
       For extra safety (don't use with lots of modules connected
    */
    void lasersOff();

    /// Scan for events
    void scanEvents(unsigned int partition, unsigned int crate, unsigned int rod, int sl = -1, 
                    bool extFlag = false, bool errorType = false);

    /// Print decoded event
    void decodeEvent(unsigned int partition, unsigned int crate, unsigned int rod, 
                     int sl, int index, bool extFlag = false, bool errorType = false);

    /**
       Decode config bit stream (only valid after SEND_CONFIG
    */
    void decodeConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                      bool skipTrim = false, bool bypass = false);

    void testLinkOutSelect(unsigned int partition, unsigned int crate, unsigned int rod,
                           unsigned int link);

    void setDebugOption(const char *opt);

    void unsetDebugOption(const char *opt);

    void listEnabledDebugOptions();

    void listDebugOptions();

    void debugStepHistogram();

    void debugContinueHistogram();

    void debugAbortHistogram();

    /* *************  TIM functs **************/

    /// Set the TIM trigger frequency
    /**
       Set both trigger and reset frequency
    */
    void timSetFrequency(unsigned int partition, unsigned int crate, 
                         double trigFreq, double rstFreq);

    /// Start the TIM generating regular triggers 
    /**
       Use currently set frequency
    */
    void freeTriggers(unsigned int partition, unsigned int crate);

    /// Stop the TIM generating regular triggers
    void stopTriggers(unsigned int partition, unsigned int crate);

    /// Tell the TIM to send an L1A 
    void timL1A(unsigned int partition, unsigned int crate);

    /// Tell TIM to send Cal + L1A
    void timCalL1A(unsigned int partition, unsigned int crate, int delay);

    /// Tell the TIM to send a softReset
    void timSoftReset(unsigned int partition, unsigned int crate);

    /// Tell the TIM to send a BCR 
    void timBCReset(unsigned int partition, unsigned int crate);

    /// Send a burst of triggers from the TIM
    void sendTimBurst(unsigned int partition, unsigned int crate, int count);

    /// Print info about the TIM
    void timVerbose(unsigned int partition, unsigned int crate);

    void timRegLoad(unsigned int partition, unsigned int crate, int reg, UINT16 val);

    UINT16 timReadRegister(unsigned int partition, unsigned int crate, int reg);

    void requestHardReset(UINT32 mid);

    void resumePolling();

    void stopPolling();
 private:
    TApi(const TApi&);
    TApi &operator=(const TApi &);

    /// The worker that implements the functions
    SctApi::SctApi *worker;

    // For CINT and ROOT
    ClassDef(TApi,1)
};

/*  /// The name tapi is defined in the shared library (it still needs initialising) */
/*  //extern TApi *tapi; */
#endif
