#ifndef SCANPOINTS_H
#define SCANPOINTS_H

#include <vector>
#include <string>
#include "Sct/Streamable.h"
#include "Sct/Exception.h"
#include "Sct/RangedVector.h"

using Sct::Streamable;
using Sct::LogicError;
using std::vector;
using std::string;

class TAxis;

namespace SctData {

/**
This class represents the points in a scan.  It contains methods to get the number of events, the number
of error events and the variable value for each point.  There are also some utility methods to get "bin"
values for use in setting histogram axes and to set a TAxis directly.
@author Matthew Palmer	
*/
class ScanPoints : public virtual Streamable{
public:
    /**Create a ScanPoints from data that is already known.
     * @param the number of points
     * @param pointer to float[nPoints]    : scan-point values
     * @param pointer to unsigned[nPoints] : numbers of events
     * @param pointer to unsigned[nPoints] : numbers of error events
     * @note does not take ownership of arguments - copies them;
     */
    ScanPoints(const unsigned nPoints, 
	       const float* points, 
	       const unsigned* nEvents, 
	       const unsigned* nErrorEvents) throw() ;
    
    /// Create an empty ScanPoints.
    ScanPoints() throw() ;

    /// Virtual destructor
    virtual ~ScanPoints() throw() {}
    
    ///	Get the number of scan points.
    unsigned getNPoints() const throw();
    
    ///	Get the value of the variable at a point. 
    double getPoint(const unsigned i) const throw(LogicError);
    
    /// Returns getPoint(i).
    double operator[] (const unsigned i) const throw(LogicError);
    
    ///	Returns an l-value which is the value of the variable at a point,
    double& operator[] (const unsigned i) throw(LogicError);
    
    ///	Get the number of events (triggers) at a point.
    unsigned getNEvents(const unsigned i) const throw(LogicError);
    
    ///	Get the number of error events at a point.
    unsigned getNErrorEvents(const unsigned i) const throw(LogicError);
    
    ///	Add a new point.  Takes the variable value, number of events and number of error events.
    void addPoint(const double point, const unsigned nEvents, const unsigned nErrorEvents) throw() ;
    
    /**
       Returns an array that contains bin lower edges for this ScanPoints <B>in ascending order</B>.
       Can be used for setting axes.
       @throw Throws an exception if number of points is <2, or points are neither ascending or descending.
    */
    double* getEdgesAscending() const throw(LogicError) ;

    /** 
	Returns true if sorted ascending, false if sorted descending, @throw LogicError if not sorted. 
    */
    bool ascending() const;

    /** Set a ROOT TAxis to the values in this ScanPoints.
	@throw Throws an exception if number of points is <2
    */
    void setAxis(TAxis& axis) const throw(LogicError);
    
    /// override
    virtual string getClassName() const throw();
    
private:
    /// internal class holds data, one per point
    class ScanPoint{
    public:
	ScanPoint(const double point, const unsigned nEv, const unsigned nErr) throw() : 
	    point(point), nEvents(nEv), nErrorEvents(nErr)  {}
	/// destructor does nothing
	~ScanPoint() throw() {}
	///@return the point as an l-value
	operator double&() throw() {return point;}
	///@return the point
	operator const double() const throw() {return point;}
	double point;              ///<The variable value
	unsigned nEvents;          ///< The number of events point.
	unsigned nErrorEvents;     ///< The number of error events.
    };

    Sct::RangedVector<ScanPoint> points;   ///< vector of ScanPoint. 
};
}
#endif //#ifndef SCANPOINTS_H
