#ifndef SCTDATA_CHIPCONFIGURATION_H
#define SCTDATA_CHIPCONFIGURATION_H

#include <string>
#include <climits>
#include <CommonWithDsp/sctStructure.h>
#include "Sct/Streamable.h"
#include "Sct/SctParameters.h"
#include "Sct/Exception.h"
#include "Sct/OutOfRangeError.h"

using Sct::LogicError;
using Sct::OutOfRangeError;
namespace SctData {
  
/**
  This class represents the configuration of a chip.
  It is based on the data stored in ABCDChip
  @author Matthew Palmer
  @date 28 May 2003
  */
class ChipConfiguration : public virtual Sct::Streamable {
public:
    ChipConfiguration(ABCDChip& chipConfig) throw();
    virtual ~ChipConfiguration() throw() {}
    virtual std::string getClassName() const throw();
    
    /// @return true if chip is participating in a scan
    bool isActive() const throw();
    /// set the chip activity status @param active if set true
    void setActive(const bool active) throw();
    /** There is a had-coded address associated with each chip position on the module.
	@return the address.
    */
    unsigned char getAddress() const throw();
    /// @return the trim target in steps of 2.5 mV
    unsigned char getTrimTarget() const throw();
    /// set the trim target @param target in steps of 2.5 mV
    void setTrimTarget(const unsigned char target) throw(LogicError);
    /// get the trim  @param channel 0->127 @return trim 0->15
    unsigned char getTrim(unsigned int channel) const throw(LogicError);
    ///set the trim @param channel 0->127 @param trim 0->15
    void setTrim(const unsigned channel, const unsigned char trim) throw(LogicError);
    /// get the value of the threshold in UNITS?
    unsigned char getThreshold() const throw();
    /// set the threhold in UNITS?
    void setThreshold(const unsigned char threshold) throw(LogicError);
    /// get the value of the calibration charge
    unsigned char getCalCharge() const throw();
    /// set the value of the calibration charge in UNITS
    void setCalCharge(const unsigned char calCharge) throw();
    /// get the value of the strobe delay in UNITS
    unsigned char getStrobeDelay() const throw();
    /// set the value of the strobe delay in UNITS
    void setStrobeDelay(const unsigned char strobeDelay) throw(LogicError);
    /// unmask all channels
    void resetMask() throw();
    /// mask mutator @param channel index 0->127 @param mask if true, unmask if false
    void setMask(const unsigned ichannel, const bool value) throw(LogicError);
    /// mask mutator @param channel index 0->127
    void mask(const unsigned ichannel) throw(LogicError);
    /// mask mutator @param channel index 0->127
    void unmask(const unsigned ichannel) throw(LogicError);
    /// mask access @return true if channel is masked @param channel index 0->127
    bool isMasked(const unsigned ichannel) const throw(LogicError);

    unsigned char getTrimRange() const throw(); 
    void setTrimRange(const unsigned char) throw(LogicError);
     /* A functional representation of the response
     * curve (equivalent threshold as a function of charge)
     * so that one may request ROD to set the threshold to
     * a value specified in fC. @{ get the index (type) of the RC function
     *  @return 0 - no calibration information
     *  1 - second order polynomial
     *  2 - "grillo" function
     *  3 - exponential
     *  4 - straight line fit */
    char getRcFunctionIndex() const throw();
    /// Response curve function index. @param 
    void setRcFunctionIndex(const char) throw(LogicError);
    ///	Get the response curve parameter @param index of parameter 0->2; @return value of parameter
    double getRcParam(const unsigned ipar) const throw(LogicError);
    /**
       Set the response curve parameter @param index of parameter 0->2;
       @param the value to set that parameter
    */
    void setRcParam(const unsigned ipar, const double val) const throw(LogicError);
    ///@}

    /**  
	 theshold calibration factor by which the nominal threshold must be multiplied
	 @{ mutator @param the factor
    */
    void setCalFactor(const float factor) throw();
    /// accessor @}
    float getCalFactor() const throw();
    
    void setMaster(bool value) throw();
    void setEnd(bool value) throw();
    void setFeedThrough(bool value) throw();
    void setInputBypass(bool value) throw();
    void setOutputBypass(bool value) throw();

    bool isMaster() const throw();
    bool isEnd() const throw();
    bool isFeedThrough() const throw();
    bool isInputBypass() const throw();
    bool isOutputBypass() const throw();

    /** Get const reference to underlying ABCDChip - used by IO */
    const ABCDChip& getConfig() const{ return config;}
    /** Get non-const reference to underlying ABCDChip - used by IO */
    ABCDChip& getConfig() { return config;}
private:
    ABCDChip& config;
};

//INLINES

   inline bool ChipConfiguration::isMasked(const unsigned ichannel)const throw(LogicError) {
       return !( config.basic.mask[ichannel/32] & (1 << ichannel%32) );
   }
    
    inline void ChipConfiguration::setMask(const unsigned ichannel, const bool value) throw(LogicError){
	if (value) { mask(ichannel); } else { unmask(ichannel); }
    }
    
    inline void ChipConfiguration::unmask(const unsigned ichannel) throw(LogicError){
	/// mask register is 128 bits implimented as UINT32 mask[4];
#ifndef NDEBUG
	if (ichannel>=Sct::nChannelChip) throw OutOfRangeError<unsigned>("ChipConfiguration::mask", __FILE__, __LINE__, ichannel,0,Sct::nChannelChip-1);
#endif
	config.basic.mask[ichannel/32] |= (1<<ichannel%32);
    }

    inline void ChipConfiguration::mask(const unsigned ichannel) throw(LogicError){
#ifndef NDEBUG
	if (ichannel>=Sct::nChannelChip) throw OutOfRangeError<unsigned>("ChipConfiguration::unmask", __FILE__, __LINE__, ichannel,0,Sct::nChannelModule-1);
#endif
	config.basic.mask[ichannel/32] &= ~(1<<ichannel%32);
    }
    
    inline void ChipConfiguration::resetMask() throw(){
	for (short i=0; i<4; ++i){
	    config.basic.mask[i]=ULONG_MAX;
	}
    }

    inline bool ChipConfiguration::isActive() const throw() {
    return config.active;
}
    
    inline void ChipConfiguration::setActive(const bool active) throw() {
	config.active = active;
    }
    
    inline unsigned char ChipConfiguration::getAddress() const throw() {
	return config.address;
    }
    
    inline unsigned char ChipConfiguration::getTrimTarget() const throw() {
	return config.target;
    }

    inline void ChipConfiguration::setTrimTarget(const unsigned char target) throw(LogicError) {
	config.target = target;
    }

    inline unsigned char ChipConfiguration::getThreshold() const throw() {
	return config.basic.vthr;
    }

    inline void ChipConfiguration::setThreshold(const unsigned char threshold) throw(LogicError) {
#ifndef NDEBUG
	if (threshold>=256) throw OutOfRangeError<unsigned>("ChipConfiguration::setThreshold", __FILE__, __LINE__, threshold,0,255);
#endif
	config.basic.vthr = threshold;
    }
    
    inline double ChipConfiguration::getRcParam(const unsigned ipar) const throw(LogicError) {
#ifndef NDEBUG
	if (ipar>=3) throw OutOfRangeError<unsigned>("ChipConfiguration::getRcParam", __FILE__, __LINE__, ipar,0,2);
#endif
	return config.caldata.rc_params[ipar];
    }
    
    inline void ChipConfiguration::setRcParam(const unsigned ipar, const double val) const throw(LogicError) {
#ifndef NDEBUG
	if (ipar>=3)  throw OutOfRangeError<unsigned>("ChipConfiguration::setRcParam", __FILE__, __LINE__, ipar,0,2);
#endif	
	config.caldata.rc_params[ipar]=val;
    }

    inline char ChipConfiguration::getRcFunctionIndex() const throw(){
	return config.caldata.rc_function;
    }

    inline void ChipConfiguration::setRcFunctionIndex(const char index) throw(LogicError){
#ifndef NDEBUG
      if (index>=5) throw OutOfRangeError<unsigned>("ChipConfiguration::setFunctionIndex", __FILE__, __LINE__, index,0,4);
#endif
	config.caldata.rc_function=index;
    }
    
    inline unsigned char ChipConfiguration::getCalCharge() const throw() {
	return config.basic.vcal;
    }
    
    inline void ChipConfiguration::setCalCharge(const unsigned char calCharge) throw() {
#ifndef NDEBUG
      if (calCharge>=256) throw OutOfRangeError<unsigned>("ChipConfiguration::setCalCharge", __FILE__, __LINE__, calCharge,0,255);
#endif
	config.basic.vcal = calCharge;
    }
    
    inline unsigned char ChipConfiguration::getStrobeDelay() const throw() {
	return config.basic.delay;
    }
    
    inline void ChipConfiguration::setStrobeDelay(const unsigned char strobeDelay) throw(LogicError) {
#ifndef NDEBUG
      if (strobeDelay>=64) throw OutOfRangeError<unsigned>("ChipConfiguration::setStrobeDelay", __FILE__, __LINE__, strobeDelay,0,63);
#endif
	config.basic.delay = strobeDelay;
    }

    inline ChipConfiguration::ChipConfiguration(ABCDChip& chipConfig) throw () : config(chipConfig) {}

    inline std::string ChipConfiguration::getClassName() const throw() {
	return "SctData::ChipConfiguration";
    }

    inline void ChipConfiguration::setCalFactor(const float factor) throw(){
	config.caldata.c_factor=factor;
    }

    inline float ChipConfiguration::getCalFactor() const throw(){
	return config.caldata.c_factor;
    }
  
  inline unsigned char ChipConfiguration::getTrim(unsigned ichannel) const throw(LogicError){
#ifndef NDEBUG
    if (ichannel>=Sct::nChannelChip) throw OutOfRangeError<unsigned>("ChipConfiguration::getTrim", __FILE__, __LINE__, ichannel,0,Sct::nChannelChip-1);
#endif
    return config.trim[ichannel];
  }
  inline void ChipConfiguration::setTrim(unsigned ichannel, unsigned char value) throw(LogicError){
#ifndef NDEBUG
    if (ichannel>=Sct::nChannelChip) throw OutOfRangeError<unsigned>("ChipConfiguration::setTrim channel", __FILE__, __LINE__, ichannel,0,Sct::nChannelChip-1);
    if (value>=16) throw OutOfRangeError<unsigned>("ChipConfiguration::setTrim value", __FILE__, __LINE__, value,0,15);
#endif
    config.trim[ichannel]=value;
  }
  
  inline void ChipConfiguration::setTrimRange(unsigned char value) throw(LogicError) {
#ifndef NDEBUG
    if (value>=4) throw OutOfRangeError<unsigned>("ChipConfiguration::setTrimRange value", __FILE__, __LINE__, value,0,3);
#endif
    config.basic.config.trimRange=value;
  }
  inline unsigned char ChipConfiguration::getTrimRange() const throw() {
    return config.basic.config.trimRange;
  }
    
    inline void ChipConfiguration::setMaster(bool value) throw(){
	config.basic.config.master = value ? 1 : 0;
    }
    inline void ChipConfiguration::setEnd(bool value) throw(){
	config.basic.config.end = value ? 1 : 0;
    }
    inline void ChipConfiguration::setFeedThrough(bool value) throw(){
	config.basic.config.feedThrough = value ? 1 : 0;
    }
    inline void ChipConfiguration::setInputBypass(bool value) throw(){
	config.basic.config.inputBypass = value ? 1 : 0;
    }
    inline void ChipConfiguration::setOutputBypass(bool value) throw(){
	config.basic.config.outputBypass = value ? 1 : 0;
    }
    
    inline bool ChipConfiguration::isMaster() const throw(){
	return config.basic.config.master;
    }
    inline bool ChipConfiguration::isEnd() const throw(){
	return config.basic.config.end;
    }
    inline bool ChipConfiguration::isFeedThrough() const throw(){
	return config.basic.config.feedThrough;
    }
    inline bool ChipConfiguration::isInputBypass() const throw(){
	return config.basic.config.inputBypass;
    }
    inline bool ChipConfiguration::isOutputBypass() const throw(){
	return config.basic.config.outputBypass;
    }
    
}

#endif //#ifndef SCTDATA_CHIPCONFIGURATION_H
