package SctData;

import Sct.*;

/**
 * Represents an contiguous element of a module : may be a chip, link, channel or some other region
 */
public class ModuleElement implements Streamable {
    /**
     * Constructor takes the first and last channel of the element.
     * 0,127 for chip 0 link 0, or perhaps 768, 895 for chip 0 link 1.  
     * @param first channel contained
     * @param last channel contained (inclusive)
     * @throw IllegalArgumentException if last<first or last>1535.
     */
    public ModuleElement(int first, int last) throws IllegalArgumentException {
        if (last >= Parameters.nChannelModule || last < first) 
            throw new IllegalArgumentException("Unable to create ModuleElement with parameters first: " + first + " last: " + last);
           
        this.first = first;
        this.last = last;
    }    
    
    /**
     * Creates a ModuleElement that represents a chip
     * @param chip the chip index
     */
    public static ModuleElement chip(int chip) {
        return new ModuleElement(chip*Parameters.nChannelChip, (chip+1)*Parameters.nChannelChip - 1);
    }
    
    /**
     * Creates a ModuleElement that represents a channel
     * @param channel the channel number
     */
    public static ModuleElement channel(int channel) {
        return new ModuleElement(channel, channel);
    }
    
    /**
     * Creates a ModuleElement that represents a link
     * @param link the link index
     */
    public static ModuleElement link(int link) {
        return new ModuleElement(link*Parameters.nChannelLink, (link+1)*Parameters.nChannelLink - 1);
    }
    
    /**
     * Creates a ModuleElement that represents a module
     */
    public static ModuleElement module() {
        return new ModuleElement(0, Parameters.nChannelModule-1);
    }
    
    /// First channel contained (0->1535)
    public int getFirst() {
        return first;
    }

    /// Last channel contained (0->1535)
    public int getLast() {
        return last;
    }

    /// @return the number of channels (=last-first+1)
    public int getNChannels() {
        return last-first + 1;
    }

    /// @return true if only contains one element
    public boolean isChannel() {
        return getNChannels() == 1;
    }

    /// @return true if the channels contained precisely match one chip
    public boolean isChip() {
        return (getFirst() % Parameters.nChannelChip == 0 && getNChannels() == Parameters.nChannelChip);
    }

    /// @return true if the channels contained precisely match one link
    public boolean isLink() {
        return (getFirst() % Parameters.nChannelLink == 0 && getNChannels() == Parameters.nChannelLink);
    }

    /// @return true if all of the channels are contained
    public boolean isModule() {
        return (getFirst() % Parameters.nChannelModule == 0 && getNChannels() == Parameters.nChannelModule);
    }
    
    /// explicit equality operator @return true if first and last are equal
    public boolean equals(Object e2) {
        if (e2 instanceof ModuleElement) {
            ModuleElement me2 = (ModuleElement)e2;
            return (this.getFirst() == me2.getFirst() && this.getLast() == me2.getLast());
        }
        return false;
    }
    
    /// @return true if this element contains all of the RHS element (or equal)
    public boolean superset(ModuleElement e2) {
        return (this.getFirst() <= e2.getFirst() && this.getLast() >= e2.getLast());
    }
    
    /// @return true if this element is contained within the RHS element (or equal)
    public boolean subset(ModuleElement e2) {
        return (this.getFirst() >= e2.getFirst() && this.getLast() <= e2.getLast());
    }

    /// @return true if this element overlaps any part of RHS element
    public boolean overlaps(ModuleElement e2) {
        return (this.getFirst()<=e2.getLast() && this.getFirst()>=e2.getFirst()) 
                || (this.getLast()<=e2.getLast() && this.getLast()>=e2.getFirst());
    }

    public String getClassName() {
        return "SctData.ModuleElement";
    }
    
    public static ModuleElement read(IStream s, ObjectManager o) throws java.io.IOException {
        int first = s.readInt("FirstChannel");
        int last  = s.readInt("LastChannel");
        return new ModuleElement(first, last);
    }
    
    public void write(OStream s, ObjectManager o) throws java.io.IOException {
        s.writeInt("FirstChannel", first, false);
        s.writeInt("LastChannel", last, false);
    }
    
    private int first;
    private int last;
}
