#ifndef SCT_STREAMERVERSIONED_IOMANAGERSTREAMERVERSIONED_H
#define SCT_STREAMERVERSIONED_IOMANAGERSTREAMERVERSIONED_H

#include "../File/IOManagerFile.h"
#include "../IoExceptions.h"
#include "../LogicErrors.h"
#include <iostream>
#include <boost/thread.hpp>

namespace Sct {
namespace Archive {
  /**
     Abstract  IOManager to be used in archival, makes use of streamer versioning. 
     When writing, the appropriate (i.e. current) streamer versions are written out. 
     When reading the file, the streamer versions are first read, 
     and then the getStreamer() method overriden to return the historic streamers.
     In write mode the most recent (default) streamers are always returned.
     Concrete versions must get a lock on the mutex, and setReadMode(true) 
     at start of read. Similarly they must get a lock on the mutex and 
     setReadMode(false) at the start of a write.
   */
class IOManagerStreamerVersioned : public IOManager {
public:
  /**
     Over-ride of getStreamer returns appropriate version of streamer
     if in read mode, otherwise default IOManager::getStreamer.
  */
  virtual Streamer& getStreamer(const string& className) const;
  /** 
   * over-ride readImpl to read classnames and versions
   */
  virtual void readImpl(IStream& in, Streamable& ob, bool bReadClassName) const;
  /** 
   * over-ride readImpl to read classnames and versions
   */
   virtual  shared_ptr<Streamable> readImpl(IStream& in) const;
  /** 
   * over-ride readImpl to read classnames and versions
   */
   virtual shared_ptr<Streamable> readImpl(IStream& in, const std::string& className) const;
  /**
   * over-ride writeImpl to write classname and version
   */
  virtual void writeImpl(OStream& out, const Streamable& ob, bool bWriteClassName) const;

   /** 
   * over-ride readImpl to read classnames and versions
   */
  virtual void readImpl(IStream& in, Streamable& ob, const std::string& className) const;
  /**
   * over-ride writeImpl to write classname and version
   */
  virtual void writeImpl(OStream& out, const Streamable& ob, const std::string& className) const;

  
 protected:
  IOManagerStreamerVersioned(); ///< Public access is through instance()
  /**
     override to write class and version to OStream 
     uses writeClassName
  */
  virtual void writeClassName(OStream&, const string& classname) const;
  /** 
      override to read version from IStream and put in map. uses readClassName
      @return classname  
  */
  virtual std::string readClassName(IStream&) const;
  /** 
      concrete implimentations must lock this mutex when doing read or write,
      so that read or write mode cannot be changed by other threads. 
   */
  boost::recursive_mutex& getMutex() const;
  /**
     set read mode to true when reading, false when writing. nothrow
  */
  void setReadMode(bool mode) const;
  /**
     get read mode - true if reading, false if writing. nothrow.
   */
  bool readMode() const;
  typedef std::map<std::string, unsigned> VersionMap;
  static VersionMap& getReadVersionMap(); ///<Map of streamer versions to use for each classname
 private:
   mutable boost::recursive_mutex m_access;  ///< access must be locked for thread-safeness.
   mutable bool m_read_mode; ///< if true will use its own verion map to get appropriate streamers.
};
}
}
#endif //SCT_STREAMERVERSIONED_IOMANAGERSTREAMERVERSIONED_H
