/*
 * OStreamFile.java
 *
 * Created on 29 October 2003, 15:17
 */

package Sct.File;

import java.io.*;
import java.nio.*;
import java.nio.channels.FileChannel;

/**
 * Writes directly to a file using a binary representation
 * Uses the NIO classes for speed, but doesn't use memory-mapped files
 * on the assumption that the file size will be small.
 * @author  Matthew Palmer
 */
public class OStreamFile implements Sct.OStream {
    
    /** Creates a new instance of OStreamFile */
    public OStreamFile(NameFile name) throws IOException {
        file = new FileOutputStream(name.getName()).getChannel();
        buffer = ByteBuffer.allocate(2*1024*1024);    //Capacity of 2Mb
        buffer.order(ByteOrder.nativeOrder());
    }
    
    
    /**
     * Creates an OStreamFile that writes to a given byte[]
     * The given array must be large enough to hold all the object being
     * written.  If not, the array will contain the last section of the data.
     * The object probably ends when the last contiguous block of zeros starts.
     * @param output the array which will be filled with the binary
     * representation of the object.
     */
    public OStreamFile(byte[] output) {
        buffer = ByteBuffer.wrap(output);
        buffer.order(ByteOrder.nativeOrder());
    }
          
    /**
     * This must be called to close the underlying stream
     */
    public void close() throws IOException {               
        if (file != null && file.isOpen()) {
            write();
            file.force(true);
            file.close();
        } else {
            //Fill remainder with 0 if there is no file - probably created with byte[]
            while (buffer.position() < buffer.capacity())
                buffer.put((byte)0);
        }
    }
    
    public void writeBoolean(String name, boolean b) throws java.io.IOException {
        checkAndFlush(1);
        if (b) buffer.put((byte)1);
        else buffer.put((byte)0);
    }
    
    public void writeBooleanArray(String name, boolean[] b) throws java.io.IOException {
        checkAndFlush(b.length + 4);
        buffer.putInt(b.length);
        for (int i=0; i<b.length; ++i) {
            if (b[i]) buffer.put((byte)1);
            else buffer.put((byte)0);
        }
    }
    
    public void writeByte(String name, byte b, boolean sign) throws java.io.IOException {
        checkAndFlush(1);
        buffer.put(b);
    }
    
    public void writeByteArray(String name, byte[] b, boolean sign) throws java.io.IOException {
        checkAndFlush(b.length + 4);
        buffer.putInt(b.length);
        buffer.put(b);
    }
    
    public void writeDouble(String name, double d) throws java.io.IOException {
        checkAndFlush(8);
        buffer.putDouble(d);
    }
    
    public void writeDoubleArray(String name, double[] d) throws java.io.IOException {
        checkAndFlush(8 * d.length + 4);
        buffer.putInt(d.length);
        buffer.position(buffer.position() + buffer.asDoubleBuffer().put(d).position() * 8);
    }
    
    public void writeFloat(String name, float f) throws java.io.IOException {
        checkAndFlush(4);
        buffer.putFloat(f);
    }
    
    public void writeFloatArray(String name, float[] f) throws java.io.IOException {
        checkAndFlush(4 * f.length + 4);
        buffer.putInt(f.length);
        buffer.position(buffer.position() + buffer.asFloatBuffer().put(f).position() * 4);
    }
    
    public void writeInt(String name, int i, boolean sign) throws java.io.IOException {
        checkAndFlush(4);
        buffer.putInt(i);
    }
    
    public void writeIntArray(String name, int[] i, boolean sign) throws java.io.IOException {
        checkAndFlush(4 * i.length + 4);
        buffer.putInt(i.length);
        buffer.position(buffer.position() + buffer.asIntBuffer().put(i).position() * 4);
    }
    
    public void writeLong(String name, long i, boolean sign) throws java.io.IOException {
        checkAndFlush(8);
        buffer.putLong(i);
    }
    
    public void writeLongArray(String name, long[] i, boolean sign) throws java.io.IOException {
        checkAndFlush(8 * i.length + 4);
        buffer.putInt(i.length);
        buffer.position(buffer.position() + buffer.asLongBuffer().put(i).position() * 8);
    }
    
    public void writeShort(String name, short s, boolean sign) throws java.io.IOException {
        checkAndFlush(2);
        buffer.putShort(s);
    }
    
    public void writeShortArray(String name, short[] s, boolean sign) throws java.io.IOException {        
        checkAndFlush(2 * s.length + 4);
        buffer.putInt(s.length);
        buffer.position(buffer.position() + buffer.asShortBuffer().put(s).position() * 2);
    }
    
    public void writeString(String name, String s) throws java.io.IOException {
        checkAndFlush(s.length() + 5);
        buffer.putInt(s.length() + 1);
        buffer.put(s.getBytes());
        buffer.put((byte) 0);
    }
    
    public void writeStringArray(String name, String[] s) throws java.io.IOException {
        checkAndFlush(4);
        buffer.putInt(s.length);
        for (int i=0; i<s.length; ++i) {
            writeString(name, s[i]);
        }
    }
    
    private void checkAndFlush(int bytes) throws IOException {
        if (buffer.remaining() < bytes) {
            write();
            buffer.clear();
        }
    }
    
    //Actually does the writing
    private void write() throws IOException {
        buffer.limit(buffer.position());
        buffer.position(0);
        if (file != null && file.isOpen()) file.write(buffer);
    }
    
    
    private FileChannel file;
    private ByteBuffer buffer;
}
