//------------------------------MdspFlashLoad------------------------------ 

/*! 
 *  @brief This is a utility program for loading the flash memory which
 *  holds the initial program for the Master DSP.
 *
 *  It prompts the user for the VME slot number of the ROD, and for the name
 *  of the binary file to load, including the path to it. The entire contents
 *  of the file are written to the MDSP flash memory.
 *
 *  @author Tom Meyer (meyer@iastate.edu)
 */

#include <iostream>
using namespace std;

#include <ctype.h>
#include <string>
#include <fstream>

#include "RodModule.h"
#include "RCCVmeInterface.h"

int main(int argc, char *argv[]) {

using namespace SctPixelRod;

  const unsigned long mapSize=0xc00040;         // VME map size 
  const long numSlaves=4;                       // Number of slaves
  std::string binFileName;                      // Name of binary file to load
  ifstream binFile;                             // Pointer to binary frile
  int fileSize;                                 // Size of binary file in bytes
  
  const unsigned long prmStart=0x01400000;
  std::string fileName(""), option;
  int safeMode = false;
  int slot = -1;
  unsigned long baseAddress;

  if (argc > 1) {
    for (int i=1; i<argc; i++) {
      option = argv[i];
      if (option[0] != '-') break;
      switch (option[1]) {
        case 'f': {
          safeMode = true;
          break;
        }
        case 's': {
          slot = atoi(option.substr(2).c_str());
          break;
        }
        default: {
          break;
        }
      }
    }
  }

// Prompt for slot number
  if (slot < 0 ) {
    cout << "Enter slot number (decimal):"; 
    cin >> slot;
    while ((slot < 1) || (slot > 21)) {
      cout << "Slot number out or range [1:21], re-enter: ";
      cin >> slot;
    }
  }
  baseAddress = slot << 24;
  
// Create VME interface
  RCCVmeInterface *vme1 = new RCCVmeInterface();
  
// Create RodModule and initialize it
  RodModule* rod0 = new RodModule(baseAddress, mapSize, *vme1, numSlaves);

  if(!safeMode) {
    try{
      rod0->initialize();
    }
    catch (HpiException &h) {
      cout << "HPI exception initialising ROD\n";
      hex(cout);
      cout << h.getDescriptor() << '\n';
      cout << "calcAddr: " << h.getCalcAddr() << ", readAddr: " << 
        h.getReadAddr() << '\n';
      dec(cout);

      cout << "Try the \"safe\" mode (-f)\n";
      return 1;
    }
    catch (BaseException &b) {
      cout << "Other exception initialising ROD\n";
      cout << b << endl;

      cout << "Try the \"safe\" mode (-f)\n";
      return 1;
    };
  } else {   // Safe mode (doesn't try to talk to MDSP (for when the MDSP is corrupt))
    try {
      unsigned long int fpgaHold = 0x40;
      unsigned long int mdspReset = 0x2;

      cerr << "Put FPGAs on reset hold\n";
      // Put FPGAs on hold
      rod0->hpiLoad(FPGA_CONTROL_REG_REL_ADDR[1], fpgaHold);
      rod0->sleep(1000);
      // Reset MDSP 
      rod0->hpiLoad(FPGA_CONTROL_REG_REL_ADDR[2], mdspReset);

      rod0->sleep(1000);

      unsigned long hpicValue = 0x00010001;
      rod0->hpiLoad(HPIC, hpicValue);

      rod0->sleep(1000);
      // mdspSingleRead/Write use HPID not HPID++ (not allowed)

      cerr << "Read from EMIF (1)\n";
      for(unsigned long addr = 0x01800000; addr < 0x0180001c; addr+=4) {
        unsigned long val = rod0->mdspSingleRead(addr);
        cerr << "0x" << hex << addr << ": 0x" << val << dec << endl;
      } 

      rod0->sleep(100);

      cerr << "Write CE space setup\n";
      rod0->mdspSingleWrite(0x01800004, 0xffff3f03);
      rod0->mdspSingleWrite(0x01800008, 0x00000040);
      rod0->mdspSingleWrite(0x01800010, 0xffff3f33);
      rod0->mdspSingleWrite(0x01800014, 0xffff3f33);

      rod0->sleep(500);

      cerr << "Read from EMIF (2)\n";

      // mdspSingleRead/Write use HPID not HPID++ (not allowed)

      for(unsigned long addr = 0x01800000; addr < 0x0180001c; addr+=4) {
        unsigned long val = rod0->mdspSingleRead(addr);
        cerr << "0x" << hex << addr << ": 0x" << val << dec << endl;
      } 
    }
    catch (BaseException &b) {
      cout << "Exception \"initing\" ROD:\n" << b << endl;
      exit(0);
    }

    rod0->sleep(3000);
  }

  cout << "Enter binary file name, including extension (""q"" to quit): ";
  cin >>  binFileName;

  if (binFileName.c_str() == "q") exit(0);
  binFile.open(binFileName.c_str(), ios::binary);
  if (!binFile.is_open()) {
    cout << "Unable to open binary file." << endl;
    exit(1);
  }
  
// Get size of file
  binFile.seekg(0, ios::end);           // go to end of file
  fileSize = binFile.tellg();          // file size given by current location
  binFile.seekg(0, ios::beg);          // go back to beginning of file
    
// Create a buffer and read file into itprmSize = fileSize
  
  UINT8 * buffer;
  try {
    buffer = new UINT8[fileSize];
  }
  catch (std::bad_alloc & ba) {
    cout << "Unable to allocate buffer for binary file." << endl;
    exit(2);
  }
  binFile.read((char *)buffer, fileSize);

  cout << "Read file of size " << fileSize << endl;
  
// write buffer to MDSP flash memory
  try {
    rod0->writeBlockToFlashHpi(prmStart, buffer, fileSize);

    cout << fileSize << " bytes written to the MDSP flash memory"<< endl;
  } catch (VmeException &v) {
    cout << "VmeException creating Writing flash." << endl;
    cout << "ErrorClass = " << v.getErrorClass() << endl;
    cout << "ErrorCode = " << v.getErrorCode() << endl; 
  } catch (RodException &r) {
    cout << r.getDescriptor() <<", " << r.getData1() << ", " << r.getData2()
	 << '\n';
  }
        
// Clean up before exiting
  delete [] buffer;
  delete rod0;
  delete vme1;

  return 0;  
}


