/************************************************************************************
 * rodConfiguration.h
 *
 *  synopsis: Describes ROD specific hardware details, e.g. is this an SCT or a pixel
 *           ROD, how many formatter FPGAs are there...
 *
 *  Damon Fasching, UW Madison                            fasching@wisconsin.cern.ch
 *  Douglas Ferguson, UW Madison   (510) 486-5230         dpferguson@lbl.gov
 *
 *  modifications/bugs    
 *   - Moved rodConfiguration #defines here (were in primParams.h)   11.04.02 dpsf
 ************************************************************************************/
#ifndef ROD_CONFIGURATION
#define ROD_CONFIGURATION

#include "processor.h"

/* Some dataflow parameters */
#define EFBS_PER_ROD               2
#define FORMATTERS_PER_EFB         4
#define FORMATTERS_PER_ROD         (FORMATTERS_PER_EFB*EFBS_PER_ROD)
#define LINKS_PER_FORMATTER        12
#define DATA_LINKS_PER_EFB         (LINKS_PER_FORMATTER*FORMATTERS_PER_EFB)
#define DATA_LINKS_PER_ROD         (DATA_LINKS_PER_EFB * EFBS_PER_ROD)
#define CTRL_LINKS_PER_ROD         (DATA_LINKS_PER_ROD / 2)
#define STROBE_DELAYS_PER_BOC      24 
 
/* number of slave DSPs */
#if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
  #define N_SLV_DSPS  4
#endif
#ifdef I_AM_SLAVE_DSP
  #define N_SLV_DSPS  1
#endif

/* 6713 phase lock loop registers & device config register; these are used by the MDSP
   while setting a Rev. E ROD's SDSP clocks. */
#define PLL_TIMEOUT 5000000  // .5 sec.

#define SDSP6713_PLLCSR    0x01b7c100
#define SDSP6713_PLLM      0x01b7c110
#define SDSP6713_PLLDIV0   0x01b7c114
#define SDSP6713_DEVCFG    0x019c0200

/* configuration data structures */
struct SlvDspConfig {
	UINT8 present;
	UINT8 commOnOff;
	UINT8 type;
	UINT8 clkSpeed;
};

typedef struct RodConfig {
	UINT32 detector;
	UINT32 boardClockInMHz;
	UINT32 DSPClockInMHz;
	UINT32 numSlaves;
	UINT32 numSlvsCommOn;
	struct SlvDspConfig slvDspConfig[N_SLV_DSPS];
} RodConfig;


/* Some modes are additive; e.g. CALIBRATION +SIMULATION. The ROD will
   return an error for those modes which are incompatible. */

/* NDT: Sets the data path according to the simulation type or link_data
        +capture,  sets the atlas running mode to true so that router is
        not allowed to apply back-pressure, and sets the link masks and
        assoc. variables according to the information stored in the 
        INIT module configuration set. Automatic inmem readout triggers an
        automatic readout of the inmems' link(s) into a MDSP buffer after
        completion of each serial stream output. */
//"dummy" mode modifiers (=> exit cfg readback, evt capture, or simulation):  
#define NOCAPTURE_MODE                    0
#define NOSIMULATION_MODE                 0
#define NORMAL_MODE                       0

//mode modifiers:
#define SIMULATION_MODE                   0x0001
#define CONFIG_READBACK_MODE              0x0002
#define INMEM_EVT_CAPTURE_MODE            0x0004
#define INMEM_AUTO_READOUT_MODE           0x0008
#define CALIBRATION_SLINK_OVERRIDE_MODE   0x0010

//ROD modes:
#define ROD_INIT_MODE            0x00010000 
#define DATA_TAKING_MODE         0x00020000 
#define CALIBRATION_MODE         0x00040000
/* Continue-mode is a dummy mode which signals the routine to
   stay in the same mode while adding/removing a mode modifier. */
#define CONTINUE_MODE            0x80000000

/* routine flags: */
#define SET_MODE        0
#define STORE_MODE      1
#define RESTORE_MODE    2
#define MODIFY_MODE     3

	typedef struct RodModeCfg {
	UINT8  sim, cfgReadback, inmemEvtCapture, autoInmemReadout;
	UINT32 rodMode;
	#if  (  defined(I_AM_MASTER_DSP) \
	      ||(defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST)) )
		UINT32 rrifCmd[2], rtrCmdStat, evtsPerL1A;
		UINT32 lutSelect, cmdMask[2][2], fmtCfg[8];
		#ifdef PIXEL_ROD
		UINT32 fmt_evtsPerL1A[8];
		#endif
	#endif
	} RodModeCfg;



#if  (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
	/* Link masks defined bitwise: set all masks on= CL on +DLP +LCFG on. Note that
	   settings within the same group are mutually exclusive; the values are set so
	   that the mask setting routine will ignore inappropriate mixtures. */

	#define DATA_LINK_MASKED  0x1  /* Dynamic modebits setting for disabled link */
	#define DATA_LINK_PLAY    0x3  /* Normal dyn. modebits setting for enabled link */
	#define DATA_LINK_SKIP    0x7  /* Sets link so that it skips over the next event */ 
	#define DATA_LINK_D1P1    0xe  /* Link will dump 1st event, play out 2nd. */ 

	#define COMMAND_LINK_OFF 0x40   
	#define COMMAND_LINK_ON  0x60   

	#define LINK_CFG_OFF     0x100   /* Link on; for initialization of ROD. */
	#define LINK_CFG_ON      0x300   /* Link off; for initialization of ROD. */

	/* Internal masks: */
	#define DATA_LINK      0x00f
	#define COMMAND_LINK   0x0f0
	#define LINK_CFG       0xf00


	/* Storage flag settings for setLinkMasks: */
	#define INIT_MASK             0
	#define SET_MASK              1
	#define UPDATE_MASK           2
	#define SWITCH_MASK           3
	#define STORE_MASK            4
	#define COMPUTE_MASK_DELTAS   5

	typedef struct FmtMask {
		UINT16 fmtCfg[FORMATTERS_PER_ROD];
		UINT32 dataLinkMask[3], unused;
	} FmtMask;

	typedef struct DynMask {
		/*                                   MB0/1 */
		UINT16 modeBits[FORMATTERS_PER_ROD][2];
	} DynMask;

	typedef struct CmdMask {
		UINT32 highMask;
		UINT32 lowMask, unused[2];
	} CmdMask;
	
	typedef struct DeltaMask {
		UINT32 modeBits;
		UINT8  cmdLine, fmtCfg, dataLinkMask, unused;
	} DeltaMask;

	#define DELTA_SP0_LO  1
	#define DELTA_SP0_HI  2
	#define DELTA_SP1_LO  4
	#define DELTA_SP1_HI  8

	#define DELTA_DFLT_MB0(fmt)  (1<<(4*(fmt) +0))
	#define DELTA_DFLT_MB1(fmt)  (1<<(4*(fmt) +1))
	#define DELTA_CRTV_MB0(fmt)  (1<<(4*(fmt) +2))
	#define DELTA_CRTV_MB1(fmt)  (1<<(4*(fmt) +3))

	#define DELTA_FMT_CFG(fmt)        (1<<(fmt))
	#define DELTA_DATA_LINK_MASK(i)   (1<<(i))


	/* The MaskConfigData structure  is used to set the command & data link masks
	   when a new group of modules is being addressed. Up to 8 'mask sets' are
	   defined; the 1st five and last two should be reserved for internal program
	   use (esp. the last two). The INIT set should be set to represent the entire
	   module set. */
	#define N_MASK_SETS   8
		#define MASK_SET_0     0       //Note: Used by internal routines.
		#define MASK_SET_1     1       //              ""
		#define MASK_SET_2     2       //              ""
		#define MASK_SET_3     3       //              ""
		#define MASK_SET_OFF   4       //Reserved.
		#define MASK_SET_INIT  5       //Should be set using a primlist on startup.
		#define MASK_SET_WORK0 6       //Work0 & 1 are general purpose.
		#define MASK_SET_WORK1 7
		#define MASK_SET_ALL   8       

	typedef struct MaskConfigData {
		CmdMask   cmdMask[2]; /* [0]= default / SP0  [1]= crtv. / SP1 */ 
		DynMask   dynMask[2];   
		FmtMask   fmtMask;
		DeltaMask deltaMask[N_MASK_SETS];
	} MaskConfigData;

#endif

/* An array of ModuleMaskData (defined in the c file) stores the mask configuration
   for each module; this is used by both the MDSP and SDSPs. */
typedef struct ModuleMaskData {
	UINT8 cmdLine;
	UINT8 unused[3];
	UINT8 dataLine[4];
} ModuleMaskData;

#endif
