/************************************************************************/
/*  ProbeCrate: based on                                                */
/*      scanvme: A program by M. Joos CERN,EP/ESS/OS 			*/
/*									*/
/*  A much cut down version of scanvme that uses some heuristics        */
/*  to guess which slots contain RODs and TIMs                          */
/*									*/
/*  The main purpose of scanvme is to scan the whole 4 GBytes		*/
/*  of VME address space for slaves present. This is done by trying 	*/
/*  to read a char, short or int. Hence, only slaves with readable 	*/
/*  registers can be found.						*/
/*									*/
/*  8.06.94   MAJO  created						*/
/* 07.03.02   MAJO  Ported to VME_RCC driver for CCT SBCs		*/
/*****C 2002 - A nickel program worth a dime*****************************/

#include <list>
#include <stdio.h>
#include <stdlib.h>
#include "rcc_error/rcc_error.h"
#include "vme_rcc/vme_rcc.h"
#include "vme_rcc/universe.h"

#define PCIMEM   0xa0000000

//prototypes
static int open_universe();
static int close_universe();
static int init_universe();

//globals
volatile universe_regs_t *uni;
static unsigned int vme_address, lsi0ctl, lsi1ctl, lsi2ctl, lsi3ctl, lsi3bs, lsi3bd, lsi3to;
static unsigned int lsi4ctl, lsi5ctl, lsi6ctl, lsi7ctl;


/*****************/
int open_universe()
/*****************/
{ 
  unsigned int ret;
  
  ret = VME_Open();
  if (ret != VME_SUCCESS)
  {
    VME_ErrorPrint(ret);
    exit(-1);
  }
  
  ret = VME_UniverseMap(&vme_address);
  if (ret != VME_SUCCESS)
  {
    VME_ErrorPrint(ret);
    exit(-1);
  }
  
  uni = (universe_regs_t *)vme_address; 
  return(0);
}



/*****************/
int init_universe()
/*****************/
{
  //disable lsi0,1,2,4,5,6,7
  lsi0ctl = uni->lsi0_ctl;
  lsi1ctl = uni->lsi1_ctl;
  lsi2ctl = uni->lsi2_ctl;
  lsi4ctl = uni->lsi4_ctl;
  lsi5ctl = uni->lsi5_ctl;
  lsi6ctl = uni->lsi6_ctl;
  lsi7ctl = uni->lsi7_ctl;
  uni->lsi0_ctl = 0;
  uni->lsi1_ctl = 0;
  uni->lsi2_ctl = 0;
  uni->lsi4_ctl = 0;
  uni->lsi5_ctl = 0;
  uni->lsi6_ctl = 0;
  uni->lsi7_ctl = 0;

  //save lsi3
  lsi3ctl = uni->lsi3_ctl;
  lsi3bs = uni->lsi3_bs;
  lsi3bd = uni->lsi3_bd;  
  lsi3to = uni->lsi3_to;
  return(0);
}


/******************/
int close_universe()
/******************/
{
  //restore lsi0,1,2,3,4,5,6,7
  uni->lsi0_ctl = lsi0ctl;
  uni->lsi1_ctl = lsi1ctl;
  uni->lsi2_ctl = lsi2ctl;
  uni->lsi4_ctl = lsi4ctl;
  uni->lsi5_ctl = lsi5ctl;
  uni->lsi6_ctl = lsi6ctl;
  uni->lsi7_ctl = lsi7ctl;

  uni->lsi3_ctl = lsi3ctl;
  uni->lsi3_bs = lsi3bs;
  uni->lsi3_bd = lsi3bd;
  uni->lsi3_to = lsi3to;

  unsigned int ret = VME_UniverseUnmap(vme_address);
  if (ret != VME_SUCCESS)
    VME_ErrorPrint(ret);

  // Update the driver
  unsigned int dummy[8];
  dummy[0] = 0; //IRQ modes are not valid

  ret = VME_Update(dummy);
  if (ret)
  {
    VME_ErrorPrint(ret);
    exit(-1);
  }  
    
  ret = VME_Close();
  if (ret != VME_SUCCESS)
    VME_ErrorPrint(ret);
  return(0);
}


/**************************/
int main(int argc, char *argv[])
/**************************/
{
  open_universe();
  init_universe();

  // This is the range checked by eg LedTest input
  for(int loop = 1; loop <= 21; loop++)
  {
    int step = 0x100000;

    // Generate a mapping in the Universe
    uni->lsi3_ctl = 0x80820000;
    uni->lsi3_bs = 0xa0000000;
    uni->lsi3_bd = 0xa1000000;
    uni->lsi3_to = 0x60000000 + (loop << 24);

    // Update the driver
    unsigned int dummy[8];
    dummy[0] = 0; //IRQ modes are not valid

    unsigned int ret = VME_Update(dummy);
    if (ret)
    {
      VME_ErrorPrint(ret);
      exit(-1);
    }  

    // Attach process to the mapping
    VME_MasterMap_t master_map;

    master_map.vmebus_address  = loop << 24;
    master_map.window_size = 0x01000000;
    master_map.address_modifier = VME_A32;
    master_map.options = 0;

    int handle;
    ret = VME_MasterMap(&master_map, &handle);
    if (ret)
    {
      VME_ErrorPrint(ret);
      exit(-1);
    }  

    int found = 0;
  
    // TIM gets only one response
    //  0x0d000000

    std::list<int> foundList;

    for (int ii = 0; ii < 0x1000000; ii += step)
    {
      unsigned int idata;
      ret = VME_ReadSafeUInt(handle, ii, &idata);

      if (!ret) {
        if(loop == 13) {
          if(ii > 0) {
            printf("Multiple (or incorrect) addresses in TIM slot\n");
          } else {
            printf("Found TIM in slot 0xd\n");
          }
        } else {
          found += 1;
          foundList.push_back(ii);
        }
      }
    }

    // ROD gets 5 responses
    //  0x0f000000
    //  0x0f200000
    //  0x0f400000
    //  0x0f600000
    //  0x0fc00000
    if(loop != 13) {
      if(found == 5) {
        printf("Found ROD in slot 0x%x\n", loop);
      } else if(found) {
        printf("Something un ROD-like in slot 0x%x!\n", loop);
        for(std::list<int>::const_iterator iter = foundList.begin();
            iter != foundList.end(); iter++) {
          printf(" %d", *iter); 
        }
        printf("\n");
      }
    }

    ret = VME_MasterUnmap(handle);
    if (ret)
      VME_ErrorPrint(ret);
  }

  close_universe();
}
