#ifndef FITTER_H
#define FITTER_H

#include <is/isinfo.h>
#include <ipc/object.h>
#include <ipc/server.h>

#include <string>
#include <memory>

#include <boost/thread.hpp>

#include "FitterI.hh"
#include "Sct/Exception.h"
#include "Sct/ConfigurationException.h"
#include "Sct/LogicErrors.h"
#include "Sct/IoExceptions.h"

class TH1;
class TF1;

using namespace Sct;

/**
 * Containing code specific to the FittingService. Uses objects from SctData and Sct.
 */
namespace SctFitter {
    class FitStrategy;
    class FitterWorkerGroup;

/**
 * Class definition of the SCT fitting service.
 * This service subscribes to an input IS server, receiving new RawScanResult objects.
 * It then fits a function (dependant on the ScanType) to chips, strips, etc.
 * and outputs a FitScanResult object to the output IS server.<BR>
 * Some status information about the Fitter is available via IPC request.
 * @author Alan Barr
 * @date 16 December 2002
 */
class Fitter: public IPCObject, public virtual FitterI_T_FitterInterface, private boost::noncopyable{
private:
    /// singleton
    Fitter(const string& fitStrategyName) throw(ConfigurationException);
    /// private destructor
    ~Fitter() throw();
    static Fitter* fitter;
    
public:
    /**
      Gets a reference to the instance of the Fitter.
      If it hasn't been initialized, it calls initialize() first
    */
    static Fitter& instance();
    
    /**
      Initialize the Fitter instance
      @param fitStrategyName the name of the FitStrategy to use.  Default is RootFitStrategy
      @throws ConfigurationException if the fitStrategyName is not a recognized FitStrategy
      */
    static Fitter& initialize(const string& fitStrategyName = "RootFitStrategy") throw(ConfigurationException);
    
    /**
     * The command to start the workers. @param nWorker the number of worker threads.
     */
    void go(unsigned nWorker) throw(IsException);
    /**
     * Get the status information
     */
    const char* status() const throw();
    /**
     * As for getStatus, but available over IPC. Makes a new char[] which ipc then deletes.
     * @param the status of the IPC request.
     */
    virtual char* status(FitterIStatus* status) throw();
    /**
     * set the fit option over IPC.
     * (asynchronous).
     * Passes string on to FitStrategy;
     * @param the status of the IPC request.
     * @param the string to set as the options e.g. "NQR" for ROOT.
     */
    void setFitOptions(const string opt) throw(LogicError) ;
    /// as for setFitOptions, but over IPC.
    virtual void setFitOptions(FitterIStatus* status, char* opt) throw() ;
    /**
     * set the fit option over IPC.
     * (asynchronous).
     * @param the status of the IPC request : failure if name not found in map.
     * @param tries to find the stragegy with this name 
     */
    virtual void setFitStrategy(FitterIStatus* status, char* strategy) throw() ;
    /**
     * Finds the FitStrategy name. Makes a new char[] which ipc then deletes.
     */
    virtual char* getFitStrategy(FitterIStatus* status) throw() ;
    // Fit all RawScanResults currently in IS.
    void fitAll(FitterIStatus* status) throw();
    /// Fit a named RawScanResult from IS.
    void fit(FitterIStatus* status, char* name) throw();
    /**
     * Finds the FitStrategy options. Makes a new char[] which ipc then deletes.
     */
    virtual char* getFitOptions(FitterIStatus* status) throw() ;
    /**
     * The infoReceiver callback function must take a static function as an argument.
     * This function simply calls doFits on the parameter or `rock', which is set 
     * to be `this' (Fitter). <BR>
     * At the moment it catches all errors since I think it should still
     * be in a stable state after most logic errors I can think of. AJB
     */
    static void doFits(ISCallbackInfo * isc) throw(IsException, LogicError);
    
    /** number of busy workers */
    long busy(FitterIStatus* status);

    /** number of jobs in the queue */
    long queueLength(FitterIStatus* status);

    /**
     *  returns the number of fits done so far;
     */
    long nFitsDone() const throw();
    /**
     *  returns the number of fits done so far over ipc;
     */
    long nFitsDone(FitterIStatus* status) throw();
    /**
     *  returns the number of fit errors so far;
     */
    long nFitErrors() const throw();

    /**
    *  returns the number of fit errors so far over ipc;
    */
    long nFitErrors(FitterIStatus* status) throw();
    /**
     * Get the name of the last scan which was fitted;
     */
    const char* lastScan() const throw();
    /**
     * Get the name of the last scan which was fitted over ipc;
     */
    char* lastScan(FitterIStatus* status) throw();
    
    /**
      Determine whether to use the analytic fit algorithm.  Default is false
      */
    void useAnalyticAlgorithm (FitterIStatus *_status, ilu_Boolean use);
        
    /**
      Return true if we're using analytic fit algorithms
      */
    bool isUsingAnalyticAlgorithm() const;
    
    /**
      Return true if we're using analytic fit algorithms
      */
    ilu_Boolean isUsingAnalyticAlgorithm(FitterIStatus *_status);

    
    /**
    Set the stragegy from a name the the FitStrategy map.
    @throw LogicError if that name is not in the map.
    */
    void setFitStrategy(string name) throw(LogicError) ;

    /// get the strategy. Throws an error if none is defined.
    FitStrategy& getFitStrategy() const throw(LogicError) ;

    /// Group of workers which do the actual fitting.
    FitterWorkerGroup* workergroup;
    
    /**
       Increments the number of fit errors by 1.
    */
    void incrementFitErrors() throw();
    
    /**
       Increments the number of fits done by 1.
    */
    void incrementFitsDone() throw();
    
    /**
      Called by the workers to add to the total amount of scan time
      */
    void scanDone(double time) throw();
    
    /**
      Called to add to IO time.  Assumes that main thread does IO and is running
      concurrently with workers.
      */
    void addIOTime(double time) throw();
    
    /**
      Algorithm may be complex!  Check actual code
      */
    double getAverageTimePerScan() const throw();
    
    /**
    Get the Fitter IPC server
    */
    static IPCServer& getFitterServer() throw();
    
private:      

    /*
     * counters:      
     */
    long m_nFitsDone;
    long m_nFitErrors;
    long m_nScans;
    double m_scanTimeTaken;
    double m_ioTimeTaken;
    ///Used to lock access to all the counters
    mutable boost::recursive_mutex counterMutex;
    
    /*
     * name of the last scan attempted.
     */
    string m_scan;
    /**
     * pointer to char containing the status: cached so that we don't have to make lots of char*'s.
     */
    mutable string m_status;
    /**
     * Pointer to the fit strategy.
     */
    FitStrategy* m_fitStrategy;
    /**
     * An InfoReveiver for RawRODData objects
     */
    std::auto_ptr<ISInfoReceiver> m_infoReceiver;

    /// update record of last scan attempted.
    void setLastScanName(const char* name){
	m_scan = name;
    }
    
    /// FitterWorker is allowed to update stuff.
    friend class FitterWorkerGroup;

}
; // end of definition of class Fitter;
}// end of namespace SctFitter
#endif // #ifndef FITTER_H
