///////////////////////////////////////////////////////////////////////////
// This file contains the class definition to issue 
// non-transition command and request data (SCT special commands)
//					
//    Created 01.04.03
//    by A. Abdesselam
///////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <string>
#include <vector>
#include <time.h>
//#include <stdlib.h>

//IS stuff
#include <is/isinfotmpl.h>
#include <is/isnamedinfo.h>
#include <is/isinfoany.h>

#include <cmdl/cmdargs.h>

using namespace std;

//this package stuff
#include "sctddc/SCTDdcCommandSender.hxx"
#include "sctddc/SCTDdcRequest.hxx"
#include "sctddc/SCTDCSCommand.hxx"

//constructor: need partition, controler list (at least one controller).
SCTDCSCommand::SCTDCSCommand(IPCPartition& partition, string** ctrlList) 
{
  //this for no-transition commands
  m_sctDdcCommandSender = new SCTDdcCommandSender(partition, ctrlList);

  //this for data request: "RunCtrl" is the name of the default IS server where DAQ issues data request
  string is_server("RunCtrl");
  m_sctDdcRequest = new SCTDdcRequest(partition, is_server);
}
SCTDCSCommand::~SCTDCSCommand() 
{
  //free memory
  delete m_sctDdcCommandSender;
  delete m_sctDdcRequest;
}

//**********************************************************************************************************
//========================================= ADDRESS HELP METHODS ===========================================
//**********************************************************************************************************

//=========================================== Methods for Channels ========================================

// these two  methods  convert PS coordinates to PVSS addresses (channel case)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSChannelAddress(int crate, int channel, string dpeName,
					   const char* direction, 
					   string& pvssAddress)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> tmp_vec1, tmp_vec2;
  tmp_vec1.push_back(dpeName);
  psToPVSSChannelAddress(crate, channel, tmp_vec1, direction, tmp_vec2);
  pvssAddress = tmp_vec2.front();
  return;
}
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSChannelAddress(int crate, int channel, std::vector<string> dpeNames, 
					   const char* direction,
					   std::vector<string>& pvssAddresses)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint addresses
  char addresses[200];
  
  // make the crate addressese
  if(crate < 10)
    {
      //add 0 to make two digit number
      sprintf(addresses, "Crate0%d", crate);
    }
  else if(crate < 100)
    {
      sprintf(addresses, "Crate%d", crate);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The crate number"<<crate<<" is not correct"<<endl;
    }

  // add the rest of the addresses: channel, parameter,... 
  if(channel < 10)
    {
      //add 0 to make two digit number
      sprintf(addresses, "%s.Channel0%d", addresses, channel);
    }
  else if(channel < 100)
    {
      sprintf(addresses, "%s.Channel%d", addresses, channel);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The channel number"<<channel<<" is not correct"
	  <<endl;
    }
  
  char addressesTmp[200];
  for(unsigned int i=0; i< dpeNames.size(); i++)
    {
      const char* c_dpeName = dpeNames[i].c_str();
      sprintf(addressesTmp, "%s.%s.%s", addresses, c_dpeName, direction);
      // put addresses in a string and then in a vector
      pvssAddresses.push_back(string(addressesTmp)); 
    }
  return;
}

//=========================================== Methods for Cards ========================================

// these two  methods  convert PS coordinates to PVSS addresses (card case)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSCardAddress(int crate, int card, string dpeName,
					const char* direction, string& pvssAddress)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> tmp_vec1, tmp_vec2;
  tmp_vec1.push_back(dpeName);
  psToPVSSCardAddress(crate, card, tmp_vec1, direction, tmp_vec2);
  pvssAddress = tmp_vec2.front();
  return;
}

//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSCardAddress(int crate, int card, std::vector<string> dpeNames,
					const char* direction,
					std::vector<string>& pvssAddresses)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint addresses
  char addresses[200];
  
  // make the crate addressese
  if(crate < 10)
    {
      //add 0 to make two digit number
      sprintf(addresses, "Crate0%d", crate);
    }
  else if(crate < 100)
    {
      sprintf(addresses, "Crate%d", crate);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The crate number"<<crate<<" is not correct"<<endl;
    }

  // add the rest of the addresses: card, parameter,...
  //first find card type
  string card_type;
  if(dpeNames.size()!=0) card_type = dpeNames[0].substr(0,2);

  if(card < 10)
    {
 
      //add 0 to make two digit number
      if(card_type == "LV")
	sprintf(addresses, "%s.LV_card0%d", addresses, card);
      else
	sprintf(addresses, "%s.HV_card0%d", addresses, card);
    }
  else if(card < 100)
    {
      if(card_type == "LV")
	sprintf(addresses, "%s.LV_card%d", addresses, card);
      else
	sprintf(addresses, "%s.HV_card%d", addresses, card);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The card number"<<card<<" is not correct"
	  <<endl;
    }
  char addressesTmp[200];
  for(unsigned int i=0; i< dpeNames.size(); i++)
    {
      const char* c_dpeName = dpeNames[i].c_str();
      sprintf(addressesTmp, "%s.%s.%s", addresses, c_dpeName, direction);
      // put addresses in a string and then in a vector
      pvssAddresses.push_back(string(addressesTmp)); 
    }
  return;
}

//======================================== Methods for Crates Controllers =================================

// these two  methods  convert PS coordinates to PVSS addresses (crate controller case)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSCrateCtrlAddress(int crate, string dpeName, const char* direction,
					 string& pvssAddress)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> tmp_vec1, tmp_vec2;
  tmp_vec1.push_back(dpeName);
  psToPVSSCrateCtrlAddress(crate, tmp_vec1, direction, tmp_vec2);
  pvssAddress = tmp_vec2.front();
  return;
}

//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSCrateCtrlAddress(int crate, std::vector<string> dpeNames,
					 const char* direction,
					 std::vector<string>& pvssAddresses)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint addresses 
  char addresses[200];
  
  // make the crate addressese
  if(crate < 10)
    {
      //add 0 to make two digit number
      sprintf(addresses, "Crate0%d.Controller", crate);
    }
  else if(crate < 100)
    {
      sprintf(addresses, "Crate%d.Controller", crate);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The crate number"<<crate<<" is not correct"<<endl;
    }

  char addressesTmp[200];
  for(unsigned int i=0; i< dpeNames.size(); i++)
    {
      const char* c_dpeName = dpeNames[i].c_str();
      sprintf(addressesTmp, "%s.%s.%s", addresses, c_dpeName, direction);
      // put addresses in a string and then in a vector
      pvssAddresses.push_back(string(addressesTmp)); 
    }
  return;
}

//======================================== Methods for Crates =========== =================================

// these two  methods  convert PS coordinates to PVSS addresses (crate case)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSCrateAddress(int crate, string dpeName, string& pvssAddress)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> tmp_vec1, tmp_vec2;
  tmp_vec1.push_back(dpeName);
  psToPVSSCrateAddress(crate, tmp_vec1, tmp_vec2);
  pvssAddress = tmp_vec2.front();
  return;
}

//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::psToPVSSCrateAddress(int crate, std::vector<string> dpeNames,
					 std::vector<string>& pvssAddresses)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint addresses
  char addresses[200];
  
  // make the crate addressese
  if(crate < 10)
    {
      //add 0 to make two digit number
      sprintf(addresses, "Crate0%d", crate);
    }
  else if(crate < 100)
    {
      sprintf(addresses, "Crate%d", crate);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The crate number"<<crate<<" is not correct"<<endl;
    }

  char addressesTmp[200];
  for(unsigned int i=0; i< dpeNames.size(); i++)
    {
      const char* c_dpeName = dpeNames[i].c_str();
      sprintf(addressesTmp, "%s.%s", addresses, c_dpeName);
      // put addresses in a string and then in a vector
      pvssAddresses.push_back(string(addressesTmp)); 
    }
}

// read from IS DCS data. the corresponding datapoint element must be declare in the 
//DDC config. file
void SCTDCSCommand::readFromIS(string dpName, float& value)
{
  m_sctDdcRequest->readFromIS(dpName, value);
  return;
}

//**********************************************************************************************************
//================================================ SOME COMMANDS ===========================================
//**********************************************************************************************************

//request IV curve for one channel
bool SCTDCSCommand::requestIVCurve(int crate, int channel, float Vstart, float Vstop, float Vstep, 
				   float Vend, float Iwarn, float Imax, int delay, int timeout)
{
  char commandPara[200];
  sprintf(commandPara, "%d|%d|%f|%f|%f|%f|%f|%f|%d", crate, channel, Vstart, Vstop, Vstep, 
	  Vend, Iwarn, Imax, delay);
  cout<< "================== commandPara = "<<commandPara<<endl;

  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "requestIVCurve";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);  
}
//request IV curve for all channel
bool SCTDCSCommand::requestIVCurveAll(int crate, float Vstart, float Vstop, float Vstep, 
				      float Vend, float Iwarn, float Imax, int delay, int timeout)
{
  char commandPara[200];
  const char* channel = "ALL";
  sprintf(commandPara, "%d|%s|%f|%f|%f|%f|%f|%f|%d", crate, channel, Vstart, Vstop, Vstep, 
	  Vend, Iwarn, Imax, delay);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "requestIVCurve";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout); 
}
//change the select line for all the channels in a given crate
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setChannelsSelectLine(int crate, int selectLine, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //command parameters 
  char commandPara[200];
  sprintf(commandPara, "%d|%d", crate, selectLine);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setChannelsSelectLine";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//transition state command for a given channel in a given crate
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::changeState(int crate, int channel, int state, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //command parameters 
  char commandPara[200];
  sprintf(commandPara, "%d|%d|%d", crate, channel, state);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "changeState";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}
//transition state command for all channels in a given crate
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::changeState(int crate, const char* allChannels, int state, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //command parameters 
  char commandPara[200];
  sprintf(commandPara, "%d|%s|%d", crate, allChannels, state);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "changeState";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//load configuration constants for all channels in a given crate
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::loadConfiguration(int crate, int state, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //command parameters 
  char commandPara[200];
  sprintf(commandPara, "%d|%d", crate, state);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "loadConfiguration";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

// reset a module (corresponding to one channel in one crate)
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::hardReset(int crate, int channel, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //command parameters 
  char commandPara[200];
  sprintf(commandPara, "%d|%d", crate, channel);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "hardReset";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

// reset all the modules (48) (all the channels in a given crate)
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::hardReset(int crate, const char* allChannels, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //command parameters 
  char commandPara[200];
  sprintf(commandPara, "%d|%s", crate, allChannels);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "hardReset";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//**********************************************************************************************************
//========================================= ACCESS CONTROL METHODS =========================================
//**********************************************************************************************************

// this is to get the control of the DCS (Arbitration procedure! realized with B. Serguey).
//---------------------------------------------------------------------------------------------------------
bool SCTDCSCommand::openDCSControl(string subSystem, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint address
  char commandPara[200];
  
  //name of the subsystem (eg SCTdcs01, SCTdcs02...)
  const char* csubSystem = subSystem.c_str();

  //some help variables
  int priority = 1;
  int priorityResponse = -99999;
  int commandResponse  = -99999;
  int counter = 0;

  //time when the access trial start
  time_t startAccessTime = time(&startAccessTime);
  //how many time the access lasts (waits)
  int deltaAccessTime = 0;
  time_t timenow = 0;

  //loop managing the access request and the waiting time
  while(priorityResponse!=priority && deltaAccessTime<timeout*10)
    {
      //counter of attemp to get DCS access
      counter++;
      // first: check the value of Hold in arbitration mechanism
      sprintf(commandPara, "%s:RequArbi.Hold|%d", csubSystem, 0);

      // this is the command name in PVSS (look at sys:DdcCommand)
      string commandName = "TakeStat";
      string commandParameters1(commandPara);

      time_t startHoldTime = time(&startHoldTime);
      int deltaHoldTime = 0;

      while(deltaHoldTime<timeout)
	{
	  // send the command to only the controller name given in the command line
	  commandResponse = m_sctDdcCommandSender->ddcSendCommand(commandName, 
							       commandParameters1, 
							       timeout);
	  if(commandResponse == 0) break;
	  //counting time for getting Hold access
	  timenow = time(&timenow);
	  deltaHoldTime = timenow - startHoldTime;
	}
      
      if(deltaHoldTime >= timeout) 
	{
	  cout << "\nWARNING: attempt number: "<< counter << endl;
	  cout << "WARNING: TimeOut to get Hold -> 0 is out, next attempt will be made\n"
	       <<endl; 
	} 
      else
	{
	  // second: set Hold equal to 1   
	  int value = 1;
	  commandResponse  = -99999;
	  sprintf(commandPara, "%s:RequArbi.Hold|%d", csubSystem, value);
	  
	  commandName = "SendStat";
	  string commandParameters2(commandPara);
	  
	  commandResponse = m_sctDdcCommandSender->ddcSendCommand(commandName, 
							       commandParameters2, timeout);
	  
	  if(commandResponse != value)
	    {
	      cerr <<"DDC ERROR: got Hold value is not equal to sent one: " << 
		commandResponse<<endl;
	      return false;
	    }
	  
	  // third: send priority request No 1
	  sprintf(commandPara, "%s:RequArbi.Req1|%d", csubSystem, priority);
	  
	  commandName = "SendRequ";
	  string commandParameters3(commandPara);
	  
	  priorityResponse = m_sctDdcCommandSender->ddcSendCommand(commandName, 
							    commandParameters3, timeout);
	  
	  if(priorityResponse != priority)
	    {
	      cout<<" \nWARNING: low priority, a next attemp will be made\n"<<endl;
	    }
	  else
	    {
	      return true;
	    }
	} //end of else and if(deltaHold...)
      
      //counting time for DCS control access
      timenow = time(&timenow);
      deltaAccessTime = timenow - startAccessTime;
    }// end of while(priorityResponse!=priority ...)
  
  
  return false;
}

// this method is to set a stat varibale (one can use also setParameter(...), see below!
//---------------------------------------------------------------------------------------------------------
bool SCTDCSCommand::setDPStatus(string dpName, int state, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint address
  char commandPara[200];
  int commandResponse  = -99999;

  const char* cdpName = dpName.c_str();
  sprintf(commandPara, "%s|%d", cdpName, state);
  
  // this is the command name in PVSS (look at sys:DdcCommand)
  string commandName = "SendStat";
  string commandParameters(commandPara);
  
  // send the command to only the controller name given in the command line
  commandResponse = m_sctDdcCommandSender->ddcSendCommand(commandName, 
						       commandParameters, timeout);
  
  if(commandResponse != state)
    {
      cerr <<"DDC ERROR: got Stat value is not equal to sent one: " << commandResponse<<endl;
      return false;
    }
  
  return true;
}

// this is a generic method, it set any parameter in the power supply PVSS project
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setParameter(int crate, int channel, string para, float value, 
				int timeout)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint address and parameter value
  char commandPara[200];
  
  // make the crate addresse
  if(crate < 10)
    {
      //add 0 to make two digit number
      sprintf(commandPara, "Crate0%d", crate);
    }
  else if(crate < 100)
    {
      sprintf(commandPara, "Crate%d", crate);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The crate number"<<crate<<" is not correct"<<endl;
    }

  // add the rest of the address: channel, parameter,... and add the value
  const char* c_para = para.c_str();
  if(channel < 10)
    {
      //add 0 to make two digit number
      sprintf(commandPara, "%s.Channel0%d.%s.Send|%f", commandPara, channel, c_para, value);
    }
  else if(channel < 100)
    {
      sprintf(commandPara, "%s.Channel%d.%s.Send|%f", commandPara, channel, c_para, value);
    }
  else
    {
      cout<< "SCTDCSCommand::setParameter: The channel number"<<channel<<" is not correct"
	  <<endl;
    }

  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setParameter";
  // put commandPara in a string
  string commandParameters(commandPara);

  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
					    commandParameters, timeout);
}

//**********************************************************************************************************
//================================================ SET METHODS =============================================
//**********************************************************************************************************

//========================================== Set methods for Channels =======================================

//set one parameter in a channel
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setChannelParameter(int crate, int channel, string dpeName, 
					  float dpeValue, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //convrtion from PS coordiante to PVSS cordinates
  string pvssAddress;
  psToPVSSChannelAddress(crate, channel, dpeName, "Send", pvssAddress);
  const char* c_pvssAddress = pvssAddress.c_str();
  //add the value to the address string
  char commandPara[200];
  sprintf(commandPara, "%s|%f", c_pvssAddress, dpeValue);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setParameters";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//set several parameter in a channel
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setChannelParameters(int crate, int channel, 
					std::map<string, float> dpeNamesAndValues, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //convrtion from PS coordiante to PVSS cordinates
  std::vector<string> pvssAddresses;
  std::vector<string> dpeNames;
  map<string, float>::const_iterator map_itr;
  for(map_itr=dpeNamesAndValues.begin(); map_itr!=dpeNamesAndValues.end(); map_itr++)
    {
      dpeNames.push_back(map_itr->first);
    }
  psToPVSSChannelAddress(crate, channel, dpeNames, "Send", pvssAddresses);
  
  //add the value to the address string
  char commandPara[8000];
  const char* c_pvssAddress;
  float dpeValue;
  int item = 0;
  for(map_itr=dpeNamesAndValues.begin(); map_itr!=dpeNamesAndValues.end(); map_itr++)
    {
      c_pvssAddress = pvssAddresses[item].c_str();
      dpeValue = map_itr->second;
      
      if(item == 0) 
	sprintf(commandPara, "%s|%f", c_pvssAddress, dpeValue);
      else
	sprintf(commandPara, "%s|%s|%f",commandPara, c_pvssAddress, dpeValue);
      
      item++;
    }
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setParameters";
  // put commandPara in a string
  string commandParameters(commandPara);
  
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//========================================== Set methods for Cards =======================================

//set one parameter in a card
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setCardParameter(int crate, int card, string dpeName, 
				    float dpeValue, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //convrtion from PS coordiante to PVSS cordinates
  string pvssAddress;
  psToPVSSCardAddress(crate, card, dpeName, "Send", pvssAddress);
  const char* c_pvssAddress = pvssAddress.c_str();
  //add the value to the address string
  char commandPara[200];
  sprintf(commandPara, "%s|%f", c_pvssAddress, dpeValue);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setParameters";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//set several parameter in a card
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setCardParameters(int crate, int card, 
				     std::map<string, float> dpeNamesAndValues, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //convrtion from PS coordiante to PVSS cordinates
  std::vector<string> pvssAddresses;
  std::vector<string> dpeNames;
  map<string, float>::const_iterator map_itr;
  for(map_itr=dpeNamesAndValues.begin(); map_itr!=dpeNamesAndValues.end(); map_itr++)
    {
      dpeNames.push_back(map_itr->first);
    }
  psToPVSSCardAddress(crate, card, dpeNames, "Send", pvssAddresses);
  
  //add the value to the address string
  char commandPara[8000];
  const char* c_pvssAddress;
  float dpeValue;
  int item = 0;
  for(map_itr=dpeNamesAndValues.begin(); map_itr!=dpeNamesAndValues.end(); map_itr++)
    {
      c_pvssAddress = pvssAddresses[item].c_str();
      dpeValue = map_itr->second;
      
      if(item == 0) 
	sprintf(commandPara, "%s|%f", c_pvssAddress, dpeValue);
      else
	sprintf(commandPara, "%s|%s|%f",commandPara, c_pvssAddress, dpeValue);
      
      item++;
    }

  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setParameters";
  // put commandPara in a string
  string commandParameters(commandPara);
  
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//======================================= Set methods for Crates Controllers =====================================

//set one parameter in a crate
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setCrateCtrlParameter(int crate, string dpeName, float dpeValue, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //convrtion from PS coordiante to PVSS cordinates
  string pvssAddress;
  psToPVSSCrateCtrlAddress(crate, dpeName, "Send", pvssAddress);
  const char* c_pvssAddress = pvssAddress.c_str();
  //add the value to the address string
  char commandPara[200];
  sprintf(commandPara, "%s|%f", c_pvssAddress, dpeValue);
  
  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setParameters";
  // put commandPara in a string
  string commandParameters(commandPara);
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}

//set several parameter in a crate
//---------------------------------------------------------------------------------------------------------
int SCTDCSCommand::setCrateCtrlParameters(int crate, std::map<string, float> dpeNamesAndValues, 
					  int timeout)
//---------------------------------------------------------------------------------------------------------
{
  //convrtion from PS coordiante to PVSS cordinates
  std::vector<string> pvssAddresses;
  std::vector<string> dpeNames;
  map<string, float>::const_iterator map_itr;
  for(map_itr=dpeNamesAndValues.begin(); map_itr!=dpeNamesAndValues.end(); map_itr++)
    {
      dpeNames.push_back(map_itr->first);
    }
  psToPVSSCrateCtrlAddress(crate, dpeNames, "Send", pvssAddresses);
  
  //add the value to the address string
  char commandPara[8000];
  const char* c_pvssAddress;
  float dpeValue;
  int item = 0;
  for(map_itr=dpeNamesAndValues.begin(); map_itr!=dpeNamesAndValues.end(); map_itr++)
    {
      c_pvssAddress = pvssAddresses[item].c_str();
      dpeValue = map_itr->second;
      
      if(item == 0) 
	sprintf(commandPara, "%s|%f", c_pvssAddress, dpeValue);
      else
	sprintf(commandPara, "%s|%s|%f",commandPara, c_pvssAddress, dpeValue);
      
      item++;
    }

  // the command Name in PVSS (data point of type DdcCommand) 
  string commandName = "setParameters";
  // put commandPara in a string
  string commandParameters(commandPara);
  
  // send the command to only the controller name given in the command line
  return  m_sctDdcCommandSender->ddcSendCommand(commandName, 
						commandParameters, timeout);
}


//---------------------------------------------------------------------------------------------------------
bool SCTDCSCommand::closeDCSControl(string subSystem, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  // PVSS DataPoint address
  char commandPara[200];

  int value = 0;
  int commandResponse  = -99999;

  const char* csubSystem = subSystem.c_str();
  sprintf(commandPara, "%s:SendStat.Req1|%d", csubSystem, value);
  
  //this is the command name in PVSS (look at sys:DdcCommand)
  string commandName = "SendStat";
  string commandParameters1(commandPara);
  
  // send the command to only the controller name given in the command line
  commandResponse = m_sctDdcCommandSender->ddcSendCommand(commandName, 
						       commandParameters1, timeout);
  
  if(commandResponse != value)
    {
      cerr<<"DDC ERROR: got Req1 value is not equal to sent one: " << commandResponse<<endl;
      return false;
    }
  
  // second: set Hold equal to 0  
  value = 0;
  sprintf(commandPara, "%s:RequArbi.Hold|%d", csubSystem, value);
  
  //string commandName = "setParameter";
  commandName = "SendStat";
  string commandParameters2(commandPara);
  
  // send the command to only the controller name given in the command line
  commandResponse = m_sctDdcCommandSender->ddcSendCommand(commandName, 
						       commandParameters2, timeout);
  
  if(commandResponse != value)
    {
      cerr<<"DDC ERROR: got Hold value is not equal to sent one: " << commandResponse<<endl;
      return false;
    }

  return true;
}


//**********************************************************************************************************
//================================================ GET METHODS =============================================
//**********************************************************************************************************

//=========================================== Get Methods for Channels ========================================

//this method reads an int channel variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getChannelParameter(int crate, int channel, string dpeName, 
					int& dpeValue, int timeout,
					const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSChannelAddress(crate, channel, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a float channel variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getChannelParameter(int crate, int channel, string dpeName, 
					float& dpeValue, int timeout,
					const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSChannelAddress(crate, channel, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a string channel variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getChannelParameter(int crate, int channel, string dpeName, 
					string& dpeValue, int timeout,
					const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSChannelAddress(crate, channel, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads several (float) channel variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getChannelParameters(int crate, int channel, std::vector<string> dpeNames, 
				std::map<string, float>& dpeValues, int timeout,
					 const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> pvssAddresses;
  psToPVSSChannelAddress(crate, channel, dpeNames, direction, pvssAddresses);
  m_sctDdcRequest->ddcSendRequest(pvssAddresses, dpeValues, timeout);
  return;
}

//=========================================== Get Methods for Cards ========================================

//this method reads an int card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCardParameter(int crate, int card, string dpeName, 
				     int& dpeValue, int timeout,
				     const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCardAddress(crate, card, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a float card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCardParameter(int crate, int card, string dpeName, 
				     float& dpeValue, int timeout,
				     const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCardAddress(crate, card, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a string card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCardParameter(int crate, int card, string dpeName, 
				     string& dpeValue, int timeout,
					const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCardAddress(crate, card, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads several (float) card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCardParameters(int crate, int card, std::vector<string> dpeNames, 
				std::map<string, float>& dpeValues, int timeout,
					const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> pvssAddresses;
  psToPVSSCardAddress(crate, card, dpeNames, direction, pvssAddresses);
  m_sctDdcRequest->ddcSendRequest(pvssAddresses, dpeValues, timeout);
  return;
}

//================================== Get Methods for Crates Controllers  ==================================

//this method reads an int card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateCtrlParameter(int crate, string dpeName, int& dpeValue, int timeout,
					  const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCrateCtrlAddress(crate, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a float card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateCtrlParameter(int crate, string dpeName, float& dpeValue, int timeout,
					  const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCrateCtrlAddress(crate, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a string card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateCtrlParameter(int crate, string dpeName, string& dpeValue, int timeout,
					  const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCrateCtrlAddress(crate, dpeName, direction, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads several (float) crate variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateCtrlParameters(int crate, std::vector<string> dpeNames, 
				std::map<string, float>& dpeValues, int timeout,
					   const char* direction)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> pvssAddresses;
  psToPVSSCrateCtrlAddress(crate, dpeNames, direction, pvssAddresses);
  m_sctDdcRequest->ddcSendRequest(pvssAddresses, dpeValues, timeout);
  return;
}

//=========================================== Get Methods for Crates ========================================

//this method reads an int card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateParameter(int crate, string dpeName, int& dpeValue, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCrateAddress(crate, dpeName, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a float card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateParameter(int crate, string dpeName, float& dpeValue, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCrateAddress(crate, dpeName, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads a string card variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateParameter(int crate, string dpeName, string& dpeValue, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress;
  psToPVSSCrateAddress(crate, dpeName, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, dpeValue, timeout);
  return;
}
//this method reads several (float) crate variables from DCS (this is an overloading for the previous ones)
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getCrateParameters(int crate, std::vector<string> dpeNames, 
				std::map<string, float>& dpeValues, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  std::vector<string> pvssAddresses;
  psToPVSSCrateAddress(crate, dpeNames, pvssAddresses);
  m_sctDdcRequest->ddcSendRequest(pvssAddresses, dpeValues, timeout);
  return;
}

//====================================== Get Methods for IV Curve Data  ===================================

//this method reads an array of float (PVSS dyn_float) from DCS 
//---------------------------------------------------------------------------------------------------------
void SCTDCSCommand::getIVCurveData(float* float_array, int& array_size, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  string pvssAddress("dyn_array_test.float_array");
  //psToPVSSCrateAddress(crate, dpeName, pvssAddress);
  m_sctDdcRequest->ddcSendRequest(pvssAddress, float_array, array_size, timeout);
  return;
}

void SCTDCSCommand::run(string** controllers)
{
  cout << endl;
  cout << "#############   Simulation   ##############" << endl;
  cout << "#######       of DAQ sending        #######" << endl;
  cout << "##    non-transition commands for DCS    ##" << endl;
  cout << "###########################################" << endl;
  
  string	       commandName;
  string	       commParameters;
  char	       timeBuf[16];
  unsigned int   timeout = 0;
  
  
  while(true) 
    {
      // Input the command name
      cout << endl << "Type the name of a command to be issued: ";
      cin >>  commandName;
      if(commandName == EXIT_COMMAND)
	return;
      
      // Input the command parameters
      cout << "Has the " << commandName << " any parameters (y/n)? ";
      cin >> commParameters;
      if((commParameters[0] == 'y') || (commParameters[0] == 'Y')) 
	{
	  cout << "Type the command parameters (p1|p2|...): ";
	  cin >>  commParameters;
	}
      else
	commParameters = "";
      
      // Input the timeout value
      cout << "Type the timeout value (sec): ";
      cin >>  timeBuf;
      timeout = atoi(timeBuf);
      
      // Input the controller(s) name to be addressed to
      string	ctrlName;
      string	ctrlPrompt = "*|" + *controllers[0];
      int i = 1;
      while(controllers[i] != 0) 
	{
	  ctrlPrompt = ctrlPrompt + '|' + *controllers[i];
	  i++;
	}
      cout << "Type the controller name to be addressed to" << endl;
      cout << "(" << ctrlPrompt << "): ";
      cin >> ctrlName;
      
      // Send the command
       m_sctDdcCommandSender->ddcSendCommand(ctrlName, commandName, commParameters, timeout); 
	 
    } //end while(true)
}

