#ifndef SCTAPI_CONFIGURATIONXMLIMPL
#define SCTAPI_CONFIGURATIONXMLIMPL

#include <utility>
#include <vector>
#include <libxml/tree.h>
#include <libxml/xpath.h>

#include <is/isinfotmpl.h>

#include "log.h"
#include "configuration.h"

#include "timings.h"

class ConfigurationXMLImpl : public SctConfiguration::Configuration {
  ConfigurationXMLImpl(const ConfigurationXMLImpl &);
  ConfigurationXMLImpl &operator=(const ConfigurationXMLImpl &);

  static ConfigurationXMLImpl *m_config;

  xmlDocPtr document;
  xmlXPathContextPtr xpathContext;

  Log log;

  std::string lastFilename;

  ISInfoDictionary *is_dict;
  bool isLookupDone;

  Timings timings;

public:
  /**
     Parse the xml configuration and do XInclude processing
     The Configuration file will be loaded from environment variable SCTDAQ_ROD_CONFIGURATION_PATH
  */
  ConfigurationXMLImpl();

  /**
     Free resources
  */
  virtual ~ConfigurationXMLImpl();

  /************* Get physical configuration ****************/

  /**
     List partition ids
     Lookup the partitions defined in the configuration and return a list of integer ID's
  */
  std::list<unsigned int> listPartitions();

  /**
     List crate ids with reference to the given partition
     Lookup the crates defined in the specified partition and return a list of integer ID's
  */
  std::list<unsigned int> listCratesInPartition(unsigned int partition);

  bool isDummyCrate(unsigned int partition, unsigned int crate);

  /**
     List the rods present in a crate
     Lookup the RODs defined in the specified crate and return a list of integer ID's
  */
  std::list<unsigned int> listRodsInCrate(unsigned int partition, unsigned int crate);

  /**
     List the MURs associated with a ROD
     Return integer id's of the MURs in the specified rod.

     FIXME:  Should be in numbered order
  */
  std::list<unsigned int> listMURSInRod(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     List the modules in the given MUR
     Return a list of serial numbers associated with the specified MUR

     FIXME:  Should be in numbered order
  */
  std::list<std::string> listModulesInMUR(unsigned int partition, unsigned int MUR);

  /**
     List the modules in the given MUR
     Return a list of serial numbers associated with the redundancy links of the specified MUR

     FIXME:  Should be in numbered order (array?)
  */
  std::list<std::string> listRModulesInMUR(unsigned int partition, unsigned int MUR);

  /**
     Return a list of serial numbers of all modules that have configuration

     This includes those that are not attached to a ROD or MUR
  */
  std::list<std::string> listAllModules();

  /**
     Return a list of serial numbers of modules that have configuration but are not attached to an MUR.
  */
  std::list<std::string> listUnusedModules();

  /************** Get configurations *****************/

  /**
     Return the configuration for the module

     Deprecated: Should take MUR argument
  */
  ABCDModule getModuleConfig(const std::string module);

  /**
     Return the group for this module
  */
  short getModuleGroup(const std::string module);

  /**
     Set the group for this module
  */
  void setModuleGroup(const std::string module, short group);

  /**
     Return the Rod configuration
     Find the configuration associated with the specified ROD.
     Return a RodConfig structure.
  */
  SctConfiguration::RodConfig getRodConfig(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Return the default slave configuration
     Find the configuration associated with the slave of the specified type
     Return a SlaveConfig structure.
  */
  SctConfiguration::SlaveConfig getDefaultSlaveConfig(char type);

  /**
     Get the BOC configuration of one of 48 channels 

     Input channels are paired as one.
  */
  SctConfiguration::BOCChannelConfig getBOCConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                                unsigned int channel);

  SctConfiguration::BOCGlobalConfig getBOCGlobalConfig(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Get the BOC fibre mappings

     An array of 48*3 chars (out, in0, in1)
  */
  char *getFibreMappings(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Return the TIM configuration
     Find the configuration associated with the specified TIM.
     Return a TimConfig structure.
  */
  SctConfiguration::TimConfig getTimConfig(unsigned int partition, unsigned int crate);

  /**
     Set fibre mappings for one channel

     Any value over 48/96 leaves current value unchanged
   */
  void setFibreMapping(unsigned int partition, unsigned int crate, unsigned int rod, 
                       unsigned int channel, unsigned int tx, unsigned int rx0, unsigned int rx1);

  /**
     Print module configuration to cout
  */
  void printModuleConfig(const ABCDModule &conf);

  SctConfiguration::MURType getMURType(unsigned int MUR);

  unsigned int getFreeMurId();

  /************** Change indexing schemes ****************/

  /**
     Translate from MUR / module(1-6) number to p/k/r/c
  */
  void translateToROD(unsigned int MUR, unsigned int module,
		      unsigned int &partition, unsigned int &crate, 
		      unsigned int &rod, unsigned int &channel);
 
  /**
     Translate from MUR / module(1-6) number to p/k/r/c

     This returns the channel to which the module referred to by MUR/module 
     is connected in its redundant configuration
  */
  void translateToRROD(unsigned int MUR, unsigned int module,
		       unsigned int &partition, unsigned int &crate, 
		       unsigned int &rod, unsigned int &channel);
 
  /**
     Translate from MUR / module(1-6) number to serial number
  */
  void translateToSN(unsigned int MUR, unsigned int module,
                     std::string &sn);

  /**
     Translate from MUR / module(1-6) number to MUR/module of redundant connection
  */
  void translateToRMUR(unsigned int MUR, unsigned int module,
                       unsigned int &RMUR, unsigned int &rmodule);

  /**
     Translate from MUR / module(1-6) number to geographical barrel system
  */
  void translateToBarrel(unsigned int MUR, unsigned int module,
                         unsigned int &barrel, unsigned int &row, int &number);

  /**
     Translate from MUR / module(1-6) number to geographical endcap system
   */
  void translateToEndcap(unsigned int MUR, unsigned int module,
                         int &disk, unsigned int &ring, unsigned int &number);

  /**
     Translate from MUR / module(1-6) number to power supply channel address
   */
  void translateToPowerSupply(unsigned int MUR, unsigned int module,
                              unsigned int &partition, 
                              unsigned int &crate, unsigned int &channel);

  /**
     Translate to MUR / module(1-6) number from p/k/r/c
  */
  void translateFromROD(unsigned int partition, unsigned int crate, 
                        unsigned int rod, unsigned int channel,
                        unsigned int &MUR, unsigned int &module);

  /**
     Translate to MUR / module(1-6) number from address of redundant module

     This returns the MUR/module of the module connected using its
     redundant link to the p/k/r/c address
  */
  void translateFromRROD(unsigned int partition, unsigned int crate,
                         unsigned int rod, unsigned int channel,
                         unsigned int &MUR, unsigned int &module);
  /**
     Translate to MUR / module(1-6) number from serial number
  */
  void translateFromSN(const std::string sn,
                       unsigned int &MUR, unsigned int &module);

  /**
     Translate from redundant MUR / module(1-6) number to MUR/module via redundant connection
  */
  void translateFromRMUR(unsigned int RMUR, unsigned int rmodule,
                         unsigned int &MUR, unsigned int &module);

  /**
     Translate to MUR / module(1-6) number from geographical barrel system
  */
  void translateFromBarrel(unsigned int barrel, unsigned int row, int number,
                           unsigned int &MUR, unsigned int &module);

  /**
     Translate to MUR / module(1-6) number from geographical endcap system
   */
  void translateFromEndcap(int disk, unsigned int ring, unsigned int number,
                           unsigned int &MUR, unsigned int &module);

  /**
     Translate from power supply channel address to MUR / module(1-6) number 
   */
  void translateFromPowerSupply(unsigned int partition, 
                                unsigned int crate, unsigned int channel, 
                                unsigned int &MUR, unsigned int &module);


  /************** Change configuration ******************/

  /**
     Update or add the configuration of a module stored in memory
     Replace the configuration for the given module with the data
     stored in the given configuration
  */
  void configureModuleFromStructure(const std::string module, const ABCDModule conf);

  /**
     Update or add the configuration of a module stored in memory
     Replace the configuration for the given module with the data
     stored in the given file (the serial number is read from the file)
  */
  void configureModuleFromFile(const std::string filename);

  /**
     Configure a ROD

     Replace the in memory configuration for the given ROD
     with the configuration provided.

     If no configuration then construct it
   */
  void configureROD(unsigned int partition, unsigned int crate, unsigned int rod, SctConfiguration::RodConfig conf);

  /**
     Configure a TIM

     Replace the in memory configuration for the given TIM
     with the configuration provided.

     If no configuration then construct it
   */
  void configureTIM(unsigned int partition, unsigned int crate, SctConfiguration::TimConfig conf);

  /**
     Set a BOC configuration
  */
  void configureBOC(unsigned int partition, unsigned int crate, unsigned int rod, const SctConfiguration::BOCGlobalConfig &conf);

  /**
     Remove all configuration.
     Very dangerous!!
   */
  void clearAll();

  /**
     Configuration creation:

     Create module (write configuration)

     Change translations

     Change MUR translations
   */

  /** 
     Name a partition (for information, any use?)
   */
  void namePartition(unsigned int partition, const std::string name);

 public:
  /**
     Set a BOC configuration 
   */
/*   void setBOCConfig(int partition, int crate, int rod, int channel, const BOCChannelConfig &conf); */
  
  /**
     Set a BOC configuration 
   */
  void configureBOCChannel(unsigned int MUR, unsigned int position, const SctConfiguration::BOCChannelConfig &conf);

  /**
     Map a module (by serial number) onto a position in an MUR
  */
  void mapModuleMUR(unsigned int MUR, unsigned int order, unsigned int RMUR, unsigned int rorder, std::string number);

  /**
     Unmap a module from a position in an MUR
  */
  void unmapModuleMUR(unsigned int MUR, unsigned int order);

  /** 
     Map an MUR
  */
  void mapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int order, unsigned int number);

  /** 
     Unmap an MUR from a ROD
  */
  void unmapRODMUR(unsigned int number);

  /**
     Map Barrel MUR
  */
  void mapBarrelMUR(unsigned int MUR, unsigned int barrel, unsigned int row, int position);

  /**
     Unmap MUR from Barrel 
  */
  void unmapBarrelMUR(unsigned int MUR);

  /** 
     Get mapping of MUR to ROD
  */
  void getMapMURROD(unsigned int MUR, unsigned int &partition, unsigned int &crate, unsigned int &rod, unsigned int &order);

  /** 
     Get mapping of ROD to MUR
  */
  void getMapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int order, unsigned int &MUR);

  /*
    Get mapping of MUR to physical structure
  */
  void getMapMURPhysical(unsigned int MUR, SctConfiguration::MURType &type, int &plane, int &section, int &position);

  /**
     Map Endcap MUR
   */
  void mapEndcapMUR(unsigned int MUR, int disk, unsigned int quadrant, unsigned int position);
 
  /**
     Unmap Endcap MUR
  */
  void unmapEndcapMUR(unsigned int MUR);

  /**
     Swap two MUR names around

     This means in regards to everything about one MUR you can use the other name
     If either doesn't exist then just one MUR is renamed
  */
  void swapMURNames(unsigned int MUR1, unsigned int MUR2);

  /**
     Map Power channel
  */
  void mapPowerChannel(unsigned int MUR, unsigned int number, unsigned int partition, unsigned int crate, unsigned int channel);

  /**
     Unmap Power channel
  */
  void unmapPowerChannel(unsigned int MUR, unsigned int number);

  /**
     Set default power supply parameter
  */
  void modifyDefaultPowerParam(std::string state, std::string name, std::string type, float value);

  /**
     Set MoPS HV card parameter
   */
  void modifyHVCardParam(unsigned int crate, unsigned int card,
                         std::string name, std::string type, float value);

  /**
     Set MoPS LV card parameter
   */
  void modifyLVCardParam(unsigned int crate, unsigned int card,
                         std::string name, std::string type, float value);

  /**
     Set MoPS channel parameter referenced by MUR/number
   */
  void modifyPowerParam(unsigned int MUR, unsigned int number,
                         std::string state, std::string name, std::string type, float value);

  /**
     Set MoPS channel parameter referenced by MoPS crate/channel
   */
  void modifyPowerParamCC(unsigned int crate, unsigned int channel,
		  std::string state, std::string name, std::string type, float value);

  /**
     Get default value of any MoPS parameter
   */
  float getDefaultPowerParam(std::string state, std::string name, std::string type);

  /**
     Get MoPS HV card parameter
  */
  float getHVCardParam(unsigned int crate, unsigned int card, std::string name, std::string type);

  /**
     Get MoPS LV card parameter
  */
  float getLVCardParam(unsigned int crate, unsigned int card, std::string name, std::string type);

  /**
     Get MoPS channel parameter referenced by MUR/number
  */
  float getPowerParam(unsigned int MUR, unsigned int number, std::string state, std::string name, std::string type);

  /**
     Get MoPS channel parameter referenced by MoPS crate/channel
   */
  float getPowerParamCC(unsigned int crate, unsigned int channel, std::string state, std::string name, std::string type);

  /**
     List power params
   */
  std::list<std::string> listDefaultPowerParams();

  /**
     Get param associated with the crate
   */
  float getCratePowerParam(unsigned int crate, std::string name);

  std::list<std::string> listDCSChannelParams();

  std::list<std::string> listDCSCardParams();

  std::list<std::string> listDCSCrateParams();

  /** 
     Load configuration from named file
   */
  void loadConfiguration(const std::string &filename);

  /************** Save configuration *********************/

  /**
     Save the current configuration

     Write configuration to disc.
     If no filename is provided a unique filename based on
     the date and time is created.

     Filename eg Conf_dump_20021210142311.xml.gz (in current directory)
  */
  void saveConfiguration(const std::string filename = "");

  /**
     Save the current configuration for a module

     Write the configuration for the module modsn to disc.
  */
  void saveModuleConfiguration(const std::string modsn, const std::string filename);

  /**
     Construct the XML for a module configuration and return as a string
  */
  std::string getModuleConfigurationString(const std::string modsn, const ABCDModule config);

  /**
     Dump power supply configuration to a file
  */
  void writePowerSupplyConfiguration(std::string filename);

 protected:
  /** Protected methods for creating internal structures */

  /**
     Create a partition 

     used by namePartition, createCrate
  */
  void createPartition(unsigned int partition);

  /**
     Create a crate 

     used by createROD
  */
  void createCrate(unsigned int partition, unsigned int crate);

  /**
     Create a ROD

     used by mapRODMUR, configureROD
   */
  void createROD(unsigned int partition, unsigned int crate, unsigned int rod);

  xmlNodePtr getModuleConfigurationNode(std::string sn, std::string reason);

  void notifyModuleConfChange();

  void notifyModuleMappingChange();

  void notifySystemStructureChange();

 private:
  /**
     Return IS dictionary 
     Find if not already created
   */
  ISInfoDictionary *getISDict();

  /**
     Update the ROD configuration stored in memory

     Replace the in memory configuration for the given ROD
     with the configuration provided.
  */
  void updateRodConfig(unsigned int partition, unsigned int crate, unsigned int rod, SctConfiguration::RodConfig conf);

  /**
     Query the configuration with an XPath expression, task is for error reporting

     Throws ConfigurationException if not exactly 1 node in result
  */
  xmlNodePtr getQueryNode(const std::string query, const std::string task);

  /**
     Query the configuration with an XPath expression, task is for error reporting
     Returns list of nodes

     Throws ConfigurationException if not exactly 1 node in result
  */
  std::list<xmlNodePtr> getQueryNodeList(const std::string query, const std::string task);

  /**
     Return contents of named attribute parsed as a float.

     Throws: ConfigurationException if no attribute
   */
  float getAttrAsFloat(xmlNodePtr node, const char *attr);

  /**
     Return contents of named attribute parsed as an int.

     Default to guessing the base

     Throws: ConfigurationException if no attribute
   */
  int getAttrAsInt(xmlNodePtr node, const char *attr, int base = 0);

  /**
     Return contents of named attribute as a standard string.

     Frees the value returned by xmlGetProp
     Throws: ConfigurationException if no attribute
   */
  std::string getAttrAsString(xmlNodePtr node, const char *attr);

  /**
     Return the first element node called name that is a child of node

     If not present throw exception
  */
  xmlNodePtr getChildNode(xmlNodePtr node, const char *name);

  /**
     Return the first element node called name that is a child of node

     If not present return new one
  */
  xmlNodePtr getOrNewChildNode(xmlNodePtr node, const char *name);

  /**
     Set the content of the named child to content

     If not present create new child node
  */
  void setOrCreateChildNodeWithContent(xmlNodePtr node, const char *name, const char *content);

  /**
     Get child of node called name containing content
   */
  xmlNodePtr getChildNodeWithContent(xmlNodePtr node, const char *name, const char *content);

  /**
     Return child with integer attribute

     Create if doesn't already exist
  */
  xmlNodePtr getOrNewChildWithIntAttr(xmlNodePtr, const char *name, const char *attr, int integer);

  /**
     Return child with string attribute

     Create if doesn't already exist
  */
  xmlNodePtr getOrNewChildWithStringAttr(xmlNodePtr parent, const char *name, const char *attr, std::string value);

  /**
     Return the first child element with the attribute equal to integer
  */
  xmlNodePtr getNodeWithIntAttr(xmlNodePtr node, const char *name, const char *attr, int integer);

  /**
     Return the first child element with the attribute equal to value
  */
  xmlNodePtr getNodeWithStringAttr(xmlNodePtr node, const char *name, const char *attr, std::string value);

  /**
     Set float attribute
  */
  void setFloatAttr(xmlNodePtr node, const char *attr, float floater);

  /**
     Set integer attribute
  */
  void setIntAttr(xmlNodePtr node, const char *attr, int integer);

  /**
     Clear attribute
  */
  void clearAttr(xmlNodePtr node, const char *attr);

  /**
     Parse the xml tree provided and return the module configuration
  */
  const ABCDModule parseModuleConfig(xmlNodePtr node);

  /**
     Parse the xml tree provided and return the ROD configuration
  */
  const SctConfiguration::RodConfig parseRodConfig(xmlNodePtr node);

  /**
     Parse the xml tree provided and return the TIM configuration
  */
  const SctConfiguration::TimConfig parseTimConfig(xmlNodePtr node);

  /**
     Replace the configuration in the given node with the data
     stored in the given configuration
  */
  xmlNodePtr replaceModuleConfig(xmlNodePtr node, const ABCDModule conf, const std::string name);

 protected:
  std::map<std::string, xmlNodePtr> serialToMurNode;
  bool serialToMurNodeBuilt;

  void buildSerialToMurNode();

  std::map<std::string, xmlNodePtr> serialToConfigNode;
  bool serialToConfigNodeBuilt;

  void buildSerialToConfigNode();
};

#endif
