#ifndef SCTROD_SCTAPI_HISTO_H
#define SCTROD_SCTAPI_HISTO_H

// Make sure USE_IS etc. are defined 
#include "config.h"

#include <map.h>

#include "utility.h"

namespace SctApi {

  /// Write a histogram to a file
  void saveHistogramToFile(scan_result_ptrs histo, std::string filename);

  /// Read the histogram data in to a file
  void readHistogramToFile(const Scan &scan, const ScanEx &ex, UINT32 mid, std::string sn, 
                           scan_result_ptrs scanData, time_t saveTime, int frame = 0);

#ifdef USE_IS
  /// Read the histogram data in to IS
  void readHistogramToIS(const Scan &scan, const ScanEx &ex,
                         UINT32 mid, scan_result_ptrs scanData);
#endif

struct ModuleMask {
  UINT32 mask0;
  UINT32 mask1;
};

/// Internal class with lots more information
/**
   All this stuff restricts a scan to one ROD
 */
class RodScanEx {
 public:
  // Module slots in use
  ModuleMask channels;

  // Module slots per Group
  ModuleMask groupChannels[8];

  // Module slots for each slave
  ModuleMask slaveChannels[4];

  /// Number of slaves
  int slaves;

  /// Which DSPs to use
  char bitFieldDSP;
};

class ScanEx {
 public:
  /// Info for individual RODs
  std::map<RodLabel, RodScanEx> rodInfo;

  typedef std::map<RodLabel, RodScanEx> RodInfoMap;

  RodScanEx &getOrNewRod(const RodLabel &l, RodScanEx defaultEx);
  const RodScanEx &getRodScanInfo(const RodLabel &l) const;
  RodScanEx &getRodScanInfo(const RodLabel &l);

  /// Which modules are in which groups
  std::vector<std::list<std::string> > groupLists;

  /// What the diagnostic reg was set to... for debug mode
  unsigned int diagnosticReg;

  std::map<std::pair<RodLabel, int>, Utility::MemoryBlock> histoMap;

  std::map<std::pair<RodLabel, int>, Utility::MemoryBlock> evCountMap;

  /// Is this a trim scan?
  int trimScan;

  // Maps

  UINT8 groupDspMap[4];
  UINT8 groupSpMap[2];
  UINT8 groupRangeMap[2];
};

class ScanControl {
 protected:
  SctApi &api;

  ScanControl(SctApi &api, int bin, int event);
  ScanControl(SctApi &api);

  virtual ~ScanControl() {}
 public:
  /** Check possible and start */
  virtual void startHistogramming() = 0;

  /** Called before starting polling */
  virtual void initialisePolling() = 0;

  /** Report on what might have gone wrong */
  virtual void reportEventErrors() = 0;

  /** Report on progress at timeout */
  virtual void reportTimeout() = 0;

  /** Move to the next bin if necessary */
  virtual void nextBin() = 0;

  /** Is the scan complete? 

      @param progressMade Is set to true if progress was made (reset the timeout)
      @param newBin Is set if the end of the current bin was reached (ie nextBin should be called)
   */
  virtual bool checkScanComplete(bool &progressMade, bool &newBin) = 0;

/*   /\** Poll *\/ */
/*   virtual void pollHistogramming(ScanEx &ex, int bins, int events, int timeout = 5) = 0; */

  /** Tidy up, read out and save histogram

      @param success reports whether the scan was successful 
  */
  virtual void finishHistogram(bool success) = 0;
};

class ScanControlRODHisto : public ScanControl {
 protected:
  std::map<RodLabel, int> lastBin;
  std::map<RodLabel, int> lastEvent;

  boost::shared_ptr<Scan> scan;
  boost::shared_ptr<ScanEx> scanEx;

  int finalBin;
  int finalTrigger;

  int sectionStartBin;
  int sectionEndBin;
  int sectionEndTrigger;

  // Store for internal timing
  time_t scanStart;

  /// Read all module's histogram data
  void readHistograms();

  /// Read histogram data
  scan_result_ptrs readHistogramData(UINT32 mid, int frame = 0);

  char* readHistogramRawData(UINT32 mid, int frame = 0);

  /// Restore modules after scanning
  void postScanModuleSetup();

  /// Get chunk of memory corresponding to histogram data on a particular SDSP
  /**
     This caches the result from sendData and allows indexing each bin individually
  */
  unsigned long *getHistogramChunk(RodLabel label, int slave, unsigned long offset, unsigned long size);

  /// Get chunk of memory corresponding to event count data on a particular SDSP
  /**
     Provides cache of data which will be the same for all modules
  */
  unsigned long *getEventCountChunk(RodLabel label, int slave);

  bool checkDebugOption(int opt);

  bool findNextSection();
 public:
  ScanControlRODHisto(SctApi &api, boost::shared_ptr<Scan> aScan, boost::shared_ptr<ScanEx> aScanEx);

  virtual ~ScanControlRODHisto() {};

  /** Check possible and start */
  void startHistogramming();

  /** Setup variables after starting histogramming? */
  void initialisePolling();

  /** Report on event errors found in slaves */
  void reportEventErrors();

  typedef std::pair<std::pair<int, int>, std::pair<int, int> > TrapBuffers;

  /** Return head and tail of iframe and xframe buffers */
  TrapBuffers getTrapBuffers(const RodLabel rlabel, int dsp);

  /** Return the processing time for the last trigger */
  unsigned int getProcTime(const RodLabel rlabel, int dsp);

  /** Report on progress at timeout */
  void reportTimeout();

  /** Move to the next bin (not necessary) */
  void nextBin();

  /** Is the scan complete? */
  bool checkScanComplete(bool &progressMade, bool &newBin);

  /** Is this ROD scan complete? */
  virtual bool checkScanCompleteROD(const RodLabel &rodLabel, bool &progressMade, bool &newBin, int &totalBin);

  /** Tidy up, read out and save histogram */
  void finishHistogram(bool success);

  std::pair<RodLabel, unsigned int> findModuleSlave(unsigned int mid);
};

class ScanControlTIMHisto : public ScanControlRODHisto {
  int binCount;

 public:
  ScanControlTIMHisto(SctApi &api, boost::shared_ptr<Scan> aScan, boost::shared_ptr<ScanEx> aScanEx);

  /** Check possible and start */
  void startHistogramming();

  /** Move to the next bin if appropriate */
  void nextBin();

  /** Is this ROD scan complete? */
  bool checkScanCompleteROD(const RodLabel &rodLabel, bool &progressMade, bool &newBin, int &totalBin);

  // Same as RODHisto
/*   /\** Tidy up, read out and save histogram *\/ */
/*   void finishHistogram(); */
};

} // Close SctApi namespace

#endif
