#include "HighLevelApi.h"
#include "XmlSummaryFile.h"
#include "ModuleData.h"
#include "SctDaqRootFile.h"
#include "Exceptions.h"
#include "Globals.h"

#include "Sct/Env.h"
#include "Sct/IoExceptions.h"
#include "Sct/ILUMarshalling.h"
#include "Sct/SctNames.h"
#include "Sct/ISProxy/IOManagerISProxy.h"
#include "Sct_SctApi/SctApiServer.h"

#include "SctData/ModuleConfiguration.h"
#include "SctData/mVThresholdVariable.h"
#include "SctData/mVfromTrimTargetThresholdVariable.h"
#include "SctData/StrobeDelayVariable.h"
#include "SctData/DefaultVariable.h"
#include "SctData/NullVariable.h"
#include "SctData/RawScanResult.h"
#include "SctData/ScanPoints.h"

#include "CalibrationController/IS/TestData.h"
#include "CommonWithDsp/ABCD/ABCDscans.h"

#include <TH1.h>
#include <TH2.h>

#include <sstream>

using namespace SctData;
using namespace Sct;
using namespace Sct::IS;
using namespace Sct::ISProxy;
using namespace std;
using namespace boost;
using namespace SctConfiguration;

namespace SctTest {

HighLevelApi::HighLevelApi() : IPCObject(SctTestApi_C_HighLevelApi_instanceName, ipcServer.get()){
    try {
	setDataFile("${SCT_DAQ_ROOT}/installed/config/TestConfig/ModuleData.txt");
    } catch (Sct::FileException& fe) {
        fe.sendToMrs(MRS_INFORMATION);
    }

    try {
	setXmlSummaryFile("${SCT_DAQ_ROOT}/installed/config/TestConfig/XmlSummary.txt");
    } catch (Sct::FileException& fe) {
        fe.sendToMrs(MRS_INFORMATION);
    }

    try {
        XmlSummaryFile file(getXmlSummaryFile());
        XmlSummaryRecord record = file.getNextRecord();
        setModuleSerial(record.serialNumber);
    } catch (Exception& e) {
        e.sendToMrs(MRS_INFORMATION);
    }
}

HighLevelApi& HighLevelApi::instance() {
    static HighLevelApi api;
    return api;
}

void HighLevelApi::setModuleSerial(string serial) {
    getConfig(serial);
    this->serial = serial;
    if (!data.dataExists(serial)) throw NoDataException("No data for module " + serial, __FILE__, __LINE__);
}

string HighLevelApi::getModuleSerial() const {
    return serial;
}

void HighLevelApi::setDataFile(string file) {
    data.load(file);
}

string HighLevelApi::getDataFile() const {
    return data.getFileName();
}

void HighLevelApi::setXmlSummaryFile(string file) {
    xmlFile = Env::substituteVariables(file);;
}

string HighLevelApi::getXmlSummaryFile() const {
    return xmlFile;
}


void HighLevelApi::getConfig(string serial) {
    try {
        ModuleConfiguration* c = new ModuleConfiguration();
        if (!config) config = shared_ptr<ModuleConfiguration>(c);
        config->getABCDModule() = configServer->getModuleConfig(serial);
    } catch (ConfigurationException& e) {
        addAndGetConfig(serial);
    }
}

void HighLevelApi::addAndGetConfig(string serial) {
    XmlSummaryFile file(getXmlSummaryFile());
    string xmlFile = "";

    while (xmlFile.length() == 0 && file.hasMoreRecords()) {
        XmlSummaryRecord xsr = file.getNextRecord();
        if (xsr.serialNumber == serial) xmlFile = xsr.xmlFile;
    }

    if (xmlFile.length() == 0) {
        throw NoDataException("No data for module " + serial, __FILE__, __LINE__);
    }

    try {
        configServer->configureModuleFromFile(xmlFile);
        config->getABCDModule() = configServer->getModuleConfig(serial);
    } catch (ConfigurationException& e) {
        throw NoDataException("Could not configure module " + serial, __FILE__, __LINE__);
    }
}

void HighLevelApi::doTest(const TestInfo& info, TestData& data, const SctData::ConfigurationVariable& var) const {
    if (!config) return;
    data.nScans = data.testPoints_size;
    data.runNumber = info.runNumber;
    data.startScanNumber = info.scanNumber;
    data.status = TestData::COMPLETED;

    publishTest(data);
    for (unsigned int i=0; i<data.nScans; ++i) {
        publishScan(info, data.startScanNumber + i, var, *config);        
    }
}

void HighLevelApi::publishTest(TestData& t) const {
    string name = SctNames::getControlDataName();
    name += ".";

    ostringstream s;
    s << "TestData." << t.runNumber << "." << t.startScanNumber;

    name += s.str();
    ISInfoDictionary& is = SctNames::getISDictionary();
    ISInfo::Status stat;

    if (is.contains(name.c_str())) {
        stat = is.update(name.c_str(), t);
    } else {
        stat = is.insert(name.c_str(), t);
    }
}

string HighLevelApi::getFileName(const TestInfo& info, unsigned int scanNumber) const{
    ostringstream filename;
    filename << info.path << "/strun" << info.runNumber << "_" << scanNumber << ".root";
    return filename.str();
}

void HighLevelApi::publishScan(const TestInfo& info, unsigned int scanNumber, const ConfigurationVariable& scan_variable, const SctData::ModuleConfiguration& config) const {
    cout << getFileName(info, scanNumber) << endl;
    SctDaqRootFile file(getFileName(info, scanNumber));

    int cycleNum = file.getCycleNum(serial);
    auto_ptr<TH1F> hist0Data = file.getHistData(cycleNum, 0);
    auto_ptr<TH1F> hist1Data = file.getHistData(cycleNum, 1);
    auto_ptr<TH1F> triggerData = file.getTriggerData();

    file.dealWithOccupancy(*hist0Data, *hist1Data, *triggerData, ConfigurationVariableIOHelper::getTypeRep(scan_variable), false);

    auto_ptr<TH2D> link0Data = file.fillData(*hist0Data);
    auto_ptr<TH2D> link1Data = file.fillData(*hist1Data);

    ResultHeader* s = new ResultHeader(scanNumber, info.runNumber, serial, scan_variable);
    ScanPoints* pts = new ScanPoints();

    for (int i=1; i<=triggerData->GetNbinsX(); ++i) {
        pts->addPoint(triggerData->GetBinCenter(i),triggerData->GetBinContent(i),0);
    }

    RawScanResult r(*s, config, *pts, *link0Data.release(), *link1Data.release());

    // Publish
    IOParamsIS params(SctNames::getEventDataName());
    IOManagerISProxy::instance().write(r, &params);    
}


//ILU methods
void HighLevelApi::setDataFile (SctTestApiStatus *_status, ilu_T_CString file) {
    try {
        setDataFile(file);
    } catch (Sct::FileException& e) {
        _status->returnCode = SctTestApi_E_NoSuchFileException;
    }
}

ilu_T_CString HighLevelApi::getDataFile (SctTestApiStatus *_status) {
    return copyStringToILU(getDataFile());
}

void HighLevelApi::setXmlSummaryFile (SctTestApiStatus *_status, ilu_T_CString file) {
    try {
        setXmlSummaryFile(file);
    } catch (Sct::FileException& e) {
        _status->returnCode = SctTestApi_E_NoSuchFileException;
    }
}

ilu_T_CString HighLevelApi::getXmlSummaryFile (SctTestApiStatus *_status) {
    return copyStringToILU(getXmlSummaryFile());
}


void HighLevelApi::setModuleSerial (SctTestApiStatus *_status, ilu_T_CString serial) {
    try {
        setModuleSerial(serial);
    } catch(Sct::FileException& e) {
        _status->returnCode = SctTestApi_E_NoSuchModuleException;
    }
}

ilu_T_CString HighLevelApi::getModuleSerial (SctTestApiStatus *_status) {
    return copyStringToILU(getModuleSerial());
}

void HighLevelApi::fullBypass(SctTestApiStatus *_status) {
    TestData test;
    test.testName="FullBypassTest";
    test.testPoints_size = 6;
    test.testPoints = new double[test.testPoints_size];
    test.testVariable = 0;
    for (unsigned index=0; index<test.testPoints_size; ++index) {
        test.testPoints[index]=4.0-0.1*index;
    }
    try {
	doTest(data.getFullBypassInfo(getModuleSerial()), test, *DefaultVariable::instance(ST_TOKEN));
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
    }
}

void HighLevelApi::timeWalk(SctTestApiStatus *_status) {
    TestData test;
    test.testName="TimeWalkTest";
    test.testPoints_size = 10;
    test.testPoints = new double[test.testPoints_size];
    test.testVariable = ST_QCAL;
    test.testPoints[0]=1.25;
    test.testPoints[1]=1.5;
    test.testPoints[2]=1.75;
    test.testPoints[3]=2.0;
    test.testPoints[4]=3.0;
    test.testPoints[5]=4.0;
    test.testPoints[6]=5.0;
    test.testPoints[7]=6.0;
    test.testPoints[8]=8.0;
    test.testPoints[9]=10.0;
    try {
	doTest(data.getTimeWalkInfo(getModuleSerial()), test, StrobeDelayVariable::instance());
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
    }
    
}

void HighLevelApi::strobeDelay(SctTestApiStatus *_status) {
    TestData test;
    test.testName="StrobeDelayTest";
    test.testPoints_size = 1;
    test.testPoints = new double[test.testPoints_size];
    test.testVariable = 0;
    test.testPoints[0] = 0;
    try {
	doTest(data.getStrobeDelayInfo(getModuleSerial()), test, StrobeDelayVariable::instance());
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
    }

}

void HighLevelApi::threePointGain(SctTestApiStatus *_status) {
    TestData test;
    test.testName="NPtGainTest";
    test.testPoints_size = 3;
    test.testPoints = new double[test.testPoints_size];
    test.testVariable = ST_QCAL;
    test.testPoints[0] = 1.50;
    test.testPoints[1] = 2.00;
    test.testPoints[2] = 2.50;
    try {
	doTest(data.get3PtGainInfo(getModuleSerial()), test, mVThresholdVariable::instance());
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
    }
}

void HighLevelApi::responseCurve(SctTestApiStatus *_status) {
    TestData test;
    test.testName="NPtGainTest";
    test.testPoints_size = 10;
    test.testPoints = new double[test.testPoints_size];
    test.testVariable = ST_QCAL;
    test.testPoints[0]=0.50;
    test.testPoints[1]=0.75;
    test.testPoints[2]=1.00;
    test.testPoints[3]=1.25;
    test.testPoints[4]=1.50;
    test.testPoints[5]=2.00;
    test.testPoints[6]=3.00;
    test.testPoints[7]=4.00;
    test.testPoints[8]=6.00;
    test.testPoints[9]=8.00;
    try {
	doTest(data.getResponseCurveInfo(getModuleSerial()), test, mVThresholdVariable::instance());
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
    }
}

void HighLevelApi::noiseOccupancy(SctTestApiStatus *_status) {
    TestData test;
    test.testName="NoiseOccupancyTest";
    test.testPoints_size = 1;
    test.testPoints = new double[test.testPoints_size];
    test.testVariable = 0;
    test.testPoints[0] = 0;
    
    try {
	TestInfo info = data.getNoiseOccupancyInfo(getModuleSerial());
	//Check if it is relative to trim target or not
	SctDaqRootFile file(getFileName(info, info.scanNumber));
	int cycleNum = file.getCycleNum(serial);
	auto_ptr<TH1F> hist0Data = file.getHistData(cycleNum, 0);
	double firstBin = hist0Data->GetYaxis()->GetBinCenter(1);
	
	if (firstBin < 0) 
	    doTest(info, test, mVfromTrimTargetThresholdVariable::instance());
	else 
	    doTest(info, test, mVThresholdVariable::instance());
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
    }
}

void HighLevelApi::trim(SctTestApiStatus *_status) {
    if (!config) return;
 
    try {
	TestInfo info = data.getTrimInfo(getModuleSerial());

	TestData test;
	test.testName="TrimRangeTest";
	test.testPoints_size = 28;
	test.testPoints = new double[28];
	test.testVariable = ST_TRIM;
	test.nScans = test.testPoints_size;
	for(unsigned int i=0; i<test.testPoints_size; i++) {
	    if (i<16) {
		test.testPoints[i]=i;
	    } else {
		test.testPoints[i]=((i%4)+1)*4-1;
	    }
	}
	
	//unsigned int index = 4;
	test.runNumber = info.runNumber;
	test.startScanNumber = info.scanNumber;
	test.status = TestData::COMPLETED;
	
	publishTest(test);
	ModuleConfiguration tempConfig(*config);
	
	for(unsigned int i=0; i<test.nScans; i++) {
	    short range, point, value;
	    if (i<16) {
		range=0;
		point=i;
		value=point;
	    } else {
		range=(i-16)/4+1;
		point=(i-16)%4;
		value=(point+1)*4-1;
	    }
	    
	    for (unsigned ichip=0; ichip<12; ++ichip) {
		tempConfig.getChipConfiguration(ichip).setTrimRange(range);
		for (unsigned ichannel=0; ichannel<128; ++ichannel) {
		    tempConfig.getChipConfiguration(ichip).setTrim(ichannel,value);
		}
	    }
	    
	    try {
		publishScan(info, test.startScanNumber + i, mVThresholdVariable::instance(), tempConfig);
	    } catch (SctTestException& e) {
		e.sendToMrs(MRS_DIAGNOSTIC);
		_status->returnCode = SctTestApi_E_NoDataException;
	    }
	}
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
	return;
    }
}

void HighLevelApi::pipeline(SctTestApiStatus *_status) {
    TestData test;
    test.testName="PipelineTest";
    test.testPoints_size = 2;
    test.testPoints = new double[test.testPoints_size];
    test.testVariable = 0;
    test.testPoints[0] = 0;
    test.testPoints[1] = 0;
    try {
	doTest(data.getPipelineInfo(getModuleSerial()), test, *DefaultVariable::instance(ST_MASK));
    } catch (Exception& e) {
	e.sendToMrs(MRS_ERROR);
	_status->returnCode = SctTestApi_E_NoDataException;
    }
}

}

