#include "ScanPoints.h"
#include "TAxis.h"
#include "Sct/LogicErrors.h"
#include <algorithm>

using Sct::LogicError;
using Sct::IllegalStateError;

namespace SctData {
    
    ScanPoints::ScanPoints(const unsigned nPoints, const float* points, 
			   const unsigned* nEvents, const unsigned* nErrorEvents) throw() : points("ScanPoints::points") {
    	    for (unsigned i=0; i<nPoints; ++i) {
	    addPoint(points[i], nEvents[i], nErrorEvents[i]);
	}
    }
    
    ScanPoints::ScanPoints() throw() : points("ScanPoints::points") {}
    
    string ScanPoints::getClassName() const throw() {
	return "SctData::ScanPoints";
    }
    
    unsigned ScanPoints::getNPoints() const throw() {
	return points.size();
    }
    
    double ScanPoints::getPoint(const unsigned i) const  throw(LogicError) {
	return points[i];
    }
    
    double ScanPoints::operator[] (const unsigned i) const  throw(LogicError) {
	return points[i];
    }
    
    double& ScanPoints::operator[] (const unsigned i)  throw(LogicError) {
	return points[i].point;
    }
    
    unsigned int ScanPoints::getNEvents(const unsigned i) const  throw(LogicError) {
	return points[i].nEvents;
    }
    
    unsigned int ScanPoints::getNErrorEvents(const unsigned i) const throw(LogicError) {
	return points[i].nErrorEvents;
    }
    
    void ScanPoints::addPoint(const double point, const unsigned events, const unsigned errorEvents) throw() {
	points.push_back( ScanPoint(point, events, errorEvents) );
    }

  bool ScanPoints::ascending() const {
    if (getNPoints()<2) return true;
    bool up=false, down=false;
    double lastPoint=getPoint(0);
    
    for (unsigned ipoint=1; ipoint<getNPoints(); ++ipoint){
      double point=getPoint(ipoint);
      if (point<lastPoint) down=true;
      if (point>lastPoint) up=true;
      if (point==lastPoint || (up && down)) throw LogicError("ScanPoints not in ascending or descending order, or has repeated points", __FILE__, __LINE__);
      lastPoint=point;
    }
    return up;
  }

//Set lowedge to points[0] + 0.5 * (points[0]- points[1]);
    double* ScanPoints::getEdgesAscending() const throw(LogicError) {
	if (points.size()<2) throw IllegalStateError("ScanPoints::getEdges() must have two points", __FILE__, __LINE__) ;
	
	double* bins = new double[points.size()+1];
	
	if (ascending()){
	  double lowedge = points[0] + 0.5 * (points[0]- points[1]);
	  bins[0] = lowedge;
	  for (unsigned int i=1; i<=points.size(); i++) {
	    bins[i] = 2*points[i-1] - bins[i-1];
	  }
	} else {
	  double lowedge = points[points.size()-1] + 0.5 * (points[points.size()-1] - points[points.size()-2]);
	  bins[0] = lowedge;
	  for (unsigned int i=1; i<=points.size(); i++) {
	    bins[i] = 2*points[ points.size()-i] - bins[i-1];
	  }
	}
	return bins;
    }
  
    void ScanPoints::setAxis(TAxis& axis) const throw(LogicError) {
	double* bins = getEdgesAscending();
	axis.Set(points.size(), bins);
	delete [] bins;
    }

} /// namespace SctData
