#ifndef RESPONSECURVE_H
#define RESPONSECURVE_H

#include "Sct/Streamable.h"
#include "Sct/Exception.h"
#include "Sct/LogicErrors.h"
#include <map>
#include <TF1.h>
#include <boost/utility.hpp>
#include <boost/shared_ptr.hpp>

using std::map;
using std::auto_ptr;
using std::string;

using namespace Sct;
using namespace boost;

namespace SctData {
    class ChipConfiguration;
/**
This interface represents the user selectable mode for fitting an N Pt Gain curve.	
*/
class ResponseCurve : public virtual Streamable {
 public:
    /**
     * Get a TF1 that is the fit function. (i.e. threshold as a function of charge)
     * @throw LogicError if TF1 can't be make.
     */
    shared_ptr<TF1> getFunction() const throw() ;
    /**
       Get a TF1 that represents the inverse function. (i.e. charge as a function of threshold)	
       @throw LogicError if TF1 can't be made;
    */
    virtual shared_ptr<TF1> getInverseFunction() const throw(LogicError) = 0;
    /// return the integer index representation of the sub-class.
    const int getIndex() const throw();
    /**
       Given a TF1 and a charge point, return the gain.
       @param charge The charge for which the gain is wanted.
       @return The gain.
    */
    virtual double getGain(const double charge) const throw() = 0;
    /**
     Method used by IO refresh to create from a general mode a mode of the appropriate type.
     @return a new object of the derived class type.
    */
    virtual auto_ptr<ResponseCurve> create() const throw() = 0 ;
    /// destructor
    virtual ~ResponseCurve() throw() {;}
    /// make deep copies:
    ResponseCurve& operator=(const ResponseCurve&) throw();
    /// make deep comparison:
    bool operator==(const ResponseCurve&)const throw();
    /// over-ride streamable here <i>once and for all</i>
    virtual string getClassName() const throw() {return "SctData::ResponseCurve";}
    /// idendifier so that IO can find the right curve
    virtual string getCurveName() const throw() {return "Unknown Response Curve";}
private:
    ResponseCurve();
    shared_ptr<TF1> ptr_function;
protected:
    mutable shared_ptr<TF1> ptr_inverse;
    /// sub-classes make themselves with a pointer to their TF1.
    ResponseCurve(auto_ptr<TF1>) throw(LogicError);
};

    ///Map used within the virtual constructor idiom.
    class ResponseCurveMap : boost::noncopyable {
    public:
	/// used by derived classes to add themselves.
	bool addToMap(const string& curvename, const int index, ResponseCurve&) throw() ;
	/// access function to the singleton
	static ResponseCurveMap& getMap() throw();
	/// @return a mode of the derived class.
	/// @param the derrived class name
	/// @throw LogicError if class dosen't exist.
	auto_ptr<ResponseCurve> get(const string& curvename) const throw(LogicError);
	/// @return a mode of the derived class.
	/// @param the derrived class integer representation.
	/// @throw LogicError if class index dosen't exist.
	auto_ptr<ResponseCurve> get(const int index) const throw(LogicError);
	/// @return the response curve corresponding to a particular chip configuration.
	auto_ptr<ResponseCurve> get(const ChipConfiguration& chip) const throw(LogicError);
	/// @return the integer index representation of the response curve of name classname
	/// @param the derrived class name
	/// @throw LogicError if class dosen't exist
	int getIndex(const string& curvename) const throw(LogicError);
    private:
	map<int, ResponseCurve*> indexMap;
	map<string, int> nameMap;
    };


/**
   This class represents a linear fit to the response curve.
   It uses the standard "pol1" ROOT function, y=[0]+[1]*x
*/
    class LinearResponseCurve : public ResponseCurve {
    public:
	LinearResponseCurve() throw(LogicError);
	virtual ~LinearResponseCurve() throw(){;}
	virtual shared_ptr<TF1> getInverseFunction() const throw(LogicError);
	virtual double getGain(const double charge) const throw();
	virtual string getCurveName() const throw() {return "LinearResponseCurve";}
	virtual auto_ptr<ResponseCurve> create() const throw();
	static bool inMap; ///< dummy member for initialization into map
	/// The function used by Root for the linear.
	static double linFn(double *x, double *par) throw();
	///It's inverse also used by root.
	static double invLinFn(double *x, double *par) throw();
    };
    
/**
   This class represents the normal exponential fit to the response curve.
   The fit function is: [2]+ [0]/(1 + exp(-x/[1])).
*/
    class ExponentialResponseCurve : public ResponseCurve {
    public:
	ExponentialResponseCurve() throw(LogicError);
	virtual ~ExponentialResponseCurve() throw(){;}
	virtual shared_ptr<TF1> getInverseFunction() const throw(LogicError);
	virtual double getGain(const double charge) const throw();
 	virtual string getCurveName() const throw() {return "ExponentialResponseCurve";}
	virtual auto_ptr<ResponseCurve> create() const throw() ;
	
	/// The function used by Root for the exponential.
	static double expFn(double *x, double *par) throw();
	///It's inverse also used by root.
	static double invExpFn(double *x, double *par) throw();

	static bool inMap; ///< dummy member for initialization into map
    };
    
    
    /**
       The fit function as suggested by Alex Grillo.
       Implemented as: [0] + [1]*x/sqrt(1+[2]*[2]*x*x)
    */
    class GrilloResponseCurve : public ResponseCurve {
    public:
	GrilloResponseCurve() throw(LogicError);
	virtual ~GrilloResponseCurve() throw() {;}
	virtual shared_ptr<TF1> getInverseFunction() const throw(LogicError);
	virtual double getGain(const double charge) const throw() ;
 	virtual string getCurveName() const throw() {return "GrilloResponseCurve";}
	virtual auto_ptr<ResponseCurve> create() const throw() ;
	static bool inMap; ///< dummy member for initialization into map
	/// The function used by Root for the exponential.
	static double grilloFn(double *x, double *par) throw();
	///It's inverse also used by root.
	static double invGrilloFn(double *x, double *par) throw();
    };
    
    /**
       A quadratic fit.  Uses the builtin ROOT function pol2, y = [0] + [1]*x + [2]*x^2
    */
    class QuadraticResponseCurve : public ResponseCurve {
    public:
	QuadraticResponseCurve() throw(LogicError);
	virtual ~QuadraticResponseCurve() throw() {;}
	virtual shared_ptr<TF1> getInverseFunction() const throw(LogicError);
	virtual double getGain(const double charge) const throw();
 	virtual string getCurveName() const throw() {return "QuadraticResponseCurve";}
	virtual auto_ptr<ResponseCurve> create() const throw();
	static bool inMap; ///< dummy member for initialization into map
	/// The function used by Root for the exponential.
	static double quadFn(double *x, double *par) throw();
	///It's inverse also used by root.
	static double invQuadFn(double *x, double *par) throw();
    };

//INLINES
    inline shared_ptr<TF1> ResponseCurve::getFunction() const throw() {
	return ptr_function;
    }

    inline const int ResponseCurve::getIndex() const throw() {
	return ResponseCurveMap::getMap().getIndex(this->getCurveName());
    }
}

#endif //#ifndef RESPONSECURVE_H
