#ifndef MODULECONFIGURATION_H
#define MODULECONFIGURATION_H

#include <vector>
#include <boost/shared_ptr.hpp>
#include <CommonWithDsp/sctStructure.h>
#include "Sct/Streamable.h"
#include "Sct/SctParameters.h"
#include "ChipConfiguration.h"

using std::vector;
using boost::shared_ptr;

namespace SctData {

/**
 *  Represents the configuration of a module.
 *  An interface around ABCDModule.
 *  It is worth noting that this class and its components define access to the data 
 *  in terms of the physical setting e.g. DAC settings.  If 
 *  a user wants to think of a variable in terms of a higher level quantity, such as charge, mV etc
 *  then he should use an appropriate strategy for performing the conversion.
 *  This is motivated by the realisation that typically DAC step sizes are only approximate
 *  so expressing the setting in terms of e.g. mV is just as meaningless as expressing it in 
 *  terms of DAC steps.  However, we have decided to stick with functions in terms of nominal 
 *  values for now.
 * @author Matthew Palmer
 */
class ModuleConfiguration : public virtual Sct::Streamable {

public:
    /**
     * Simple constructor - creates an empty ABCDModule
     */
    ModuleConfiguration() throw();
    virtual ~ModuleConfiguration() throw();
    
    /**
      Copy constructor
      */
    ModuleConfiguration(const ModuleConfiguration& config) throw();
    
    /**
      Assignment operator
      */
    ModuleConfiguration& operator=(const ModuleConfiguration& config) throw();

    //Override
    virtual std::string getClassName() const throw();
    
    //Data access methods
    ChipConfiguration& getChipConfiguration(const unsigned int chip) throw();
    const ChipConfiguration& getChipConfiguration(const unsigned int chip) const throw();

    /**
       presence means module configurations are present in the
       ROD and therefore sent triggers and read out by the histogramming code
       @return true if present 
    */ 
    bool isPresent() const throw();
    /**
       active means a module "takes part in a scan" ie has
       its configuration modified by the histogramming engine
       @{ check activity @return true if active
    */
    bool isActive() const throw();
    /// set the activity @}
    void setActive(bool active) throw();
    /// 
    bool usingPrimaryClock() const throw();
    void setUsingPrimaryClock(bool usePrimaryClock) throw();

    ///@{ check if a channel is masked. @param ichannel in the range 0 to 1535.
    bool channelIsMasked(const unsigned int ichannel) const throw();    
    /// mask a particular channel. @param ichannel in the range 0 to 1535
    void maskChannel(const unsigned int ichannel) throw();
    ///@}
    
    /**
       Group ID This is used to indicate
       which slave DSP will receive the module's data (if group
       based distribution is set), and also to allow different
       module groups to be triggered independently (for
       cross-talk studies). valid range: [0,3]
       @{
       Accessor
    */
    char getGroupId() const throw();
    /// Mutator @}
    void setGroupId(const char) throw();
    
    /**
      Get const access to the underlying ABCDModule
      */
    const ABCDModule& getABCDModule() const;
    
    /**
      Get full access to the underlying ABCDModule
      */
    ABCDModule& getABCDModule();
private:
    ABCDModule configuration;				       ///< The underlying data representation
    vector<shared_ptr<ChipConfiguration> > chipConfigs;        ///< ChipConfigurations for each chip
}; // end ModuleConfiguration


    // INLINES

    inline bool ModuleConfiguration::isPresent() const throw() {
	return configuration.present == 0 ? false : true;
    }
    
    inline bool ModuleConfiguration::isActive() const throw() {
	return configuration.active == 0 ? false : true;
    }
    
    inline void ModuleConfiguration::setActive(bool active) throw() {
	configuration.active = active ? 1 : 0;
    }
    
    inline bool ModuleConfiguration::usingPrimaryClock() const throw() {
	return configuration.select == 0 ? false : true;
    }
    
    inline void ModuleConfiguration::setUsingPrimaryClock(bool usePrimaryClock) throw() {
	configuration.select = usePrimaryClock ? 1 : 0;
    }

    inline char ModuleConfiguration::getGroupId() const throw(){
	return configuration.groupId;
    }

    inline void ModuleConfiguration::setGroupId(const char id) throw(){
	configuration.groupId=id;
    }
}
#endif //#ifndef MODULECONFIGURATION_H

