#include "IOManager.h"
#include "Streamer.h"
#include "Streamable.h"
#include "Serializable.h"
#include "IStream.h"
#include "OStream.h"
#include "SctNames.h"
#include "LogicErrors.h"
#include <sstream>

using namespace std;

namespace Sct {

typedef map<string, shared_ptr<Streamer> > StreamerMap;
    
bool IOManager::addToMap(const string& className, std::auto_ptr<Streamer> auto_s) throw() {
    shared_ptr<Streamer> s (auto_s);
    const unsigned version=s->getVersion();
    string mapName = getMapName(className, version);
    //cout << "Adding ... " << mapName << endl;
    StreamerMap& default_map = getMap();
    StreamerMap& full_map = getFullVersionMap();
    
    if (full_map.find(mapName) != full_map.end()){
	string message = "IoManager:: WARNING duplicate entry for streamer/version = ";
	message += mapName;
	Sct::LogicError e(message, __FILE__, __LINE__);
	e.sendToMrs();
	ostringstream oss;		
	oss << "Known streamers:" << endl;
	for (StreamerMap::const_iterator it=full_map.begin();
	     it!=full_map.end();
	     ++it){
	    oss << (*it).first << endl; 
	}
	string diag = oss.str();
	SctNames::Mrs() << "SCT_IO" << MRS_TEXT(diag) << MRS_DIAGNOSTIC << ENDM;
	return false;
    } else {
 	full_map[mapName]=s;
        // std::cout << "adding " << mapName << s << std::endl; 
	// look for other versions of streamer for this class
	StreamerMap::iterator previous = default_map.find(className);
        if (previous==default_map.end() || (*previous).second->getVersion() < version){
	   // there is an older one in the default map
           default_map[className]=s;
	}

    }
    return true;
}


StreamerMap& IOManager::getFullVersionMap() throw(){
    static StreamerMap full_map;
    return full_map;
}

StreamerMap& IOManager::getMap() throw() {
    static StreamerMap default_map;
    return default_map;
}

Streamer& IOManager::getStreamer(const string& className) const {
    const StreamerMap& default_map = getMap();
    if (default_map.find(className) == default_map.end()) {
	ostringstream oss;		
	oss << "Known streamers:" << endl;
	for (StreamerMap::const_iterator it=default_map.begin();
	     it!=default_map.end();
	     ++it){
	    oss << (*it).first << endl; 
	}
	string diag = oss.str();
	SctNames::Mrs() << "SCT_IO" << MRS_TEXT(diag) << MRS_DIAGNOSTIC << ENDM;
	throw NoSuchStreamerException(className, "IOManager can't find a Streamer", __FILE__, __LINE__);
    }
    return *(*default_map.find(className)).second;
}

  Streamer& IOManager::getStreamer(const string& className, const unsigned version) const {
   const StreamerMap& full_map = getFullVersionMap();
   std::string mapName = getMapName(className, version);

   StreamerMap::const_iterator it = full_map.find(mapName);
   if (it==full_map.end()){
       ostringstream oss;
       oss << "Known streamer/versions:" << endl;
       for (StreamerMap::const_iterator it=full_map.begin();
            it!=full_map.end();
            ++it){
            oss << (*it).first << endl;
       }
       string diag = oss.str();
			       
       SctNames::Mrs() << "SCT_IO" << MRS_TEXT(diag) << MRS_DIAGNOSTIC << ENDM;
       throw NoSuchStreamerException(className,string("IOManager can'f find a Streamer")+mapName, __FILE__, __LINE__);      
   } else {
      return *(*it).second;
   }
  }


std::string IOManager::getMapName(const std::string& classname, const unsigned version) throw() {
    ostringstream oss;
    oss << classname << "/" << version;
    return oss.str();
}

void IOManager::writeImpl(OStream& out, const Streamable& ob, bool bWriteClassName) const {
    Streamer& s = getStreamer(ob.getClassName());
    if (bWriteClassName) writeClassName(out, ob.getClassName());
    s.write(out, ob, *this);
}

void IOManager::writeImpl(OStream& out, const Streamable& ob, const char* className) const {
  writeImpl(out, ob, string(className) );
}

void IOManager::writeImpl(OStream& out, const Streamable& ob, const string& className) const {
  Streamer& s = getStreamer(className);
  s.write(out, ob, *this);
}

shared_ptr<Streamable> IOManager::readImpl(IStream& in) const {
    string className = readClassName(in);
    Streamer& s = getStreamer(className);
    return s.read(in, *this);
}

void IOManager::readImpl(IStream& in, Streamable& ob, const string& className) const{
  Streamer& s = getStreamer(className);
  s.read(in, ob, *this);
}

shared_ptr<Streamable> IOManager::readImpl(IStream& in, const string& className) const {
    Streamer& s = getStreamer(className);
    return s.read(in, *this);    
}

void IOManager::readImpl(IStream& in, Streamable& ob, const char* className) const{
  readImpl(in, ob, string(className));
}

shared_ptr<Streamable> IOManager::readImpl(IStream& in, const char* className) const {
  return readImpl(in, string(className));
}


void IOManager::readImpl(IStream& in, Streamable& ob, bool bReadClassName) const {
    if (bReadClassName) {
	string className = readClassName(in);
	if (className != ob.getClassName()) {
	    string str = "Can't read: ";
	    str += ob.getClassName();
	    str += " from stream because stream contains a: ";
	    str += className;
	    throw StreamCorruptedException(str, __FILE__, __LINE__);
	}
    }
    Streamer& s = getStreamer(ob.getClassName());
    s.read(in, ob, *this);
}

void IOManager::writeClassName(OStream& out, const string& className) const {
    out << "SctClassName:" + className;
}

string IOManager::readClassName(IStream& in) const {
    string className = "";
    in >> className;
    if (className.find("SctClassName:") == 0) {
	return className.substr(13);
    } 
    throw StreamCorruptedException("Read string but it wasn't a className.  Got: `" + className + "'", __FILE__, __LINE__);
}

    

}
