/************************************************************************************
 * primParams.h
 *
 *  synopsis: Defines primitive ids which are used to index the array of pointers to
 *           primitive functions, structure tags for primitive data and reply data,
 *           and prototypes of primitive functions.
 *
 *  related files:
 *   listManager.c: Routines which manage the execution of a primitive list and
 *                 writing the reply data.
 *   primFuncts.c: Contains primitive execution functions common to the master and the
 *                slaves and the function which initializes the primParameters array
 *                of structures.
 *   masterPrimFuncts.c/slavePrimFuncts.c: Contain primitive execution functions
 *                specific to the master/slave DSP.
 *   primFuncts.h: Declares the structure tag which communicates necessary information
 *                to primitive functions.
 *
 *  Damon Fasching, UW Madison                            fasching@wisconsin.cern.ch
 *  Douglas Ferguson, UW Madison   (510) 486-5230         dpferguson@lbl.gov
 *
 *  modifications/bugs
 *   - Split the primitive IDs into 3 sets, so that future primitives can
 *     be placed in the appropriate section (common/slave/master) without
 *     affecting the already existing primitive lists on the host machines.
 *     Currently primitives must be added to the end of the entire list to
 *     avoid ID changes of existing primitives & thus re-writes of the
 *     primitive lists. Updated the example.                         10.04.02 dpsf
 *   - Moved rodConfiguration #defines to rodConfiguration.h, where
 *     they belong.                                                  11.04.02 dpsf
 *
 ************************************************************************************/
#ifndef PRIM_PARAMS
#define PRIM_PARAMS

#include "processor.h"
#include "utilities.h"
#include "serialStreams.h"
#include "scanControl.h"
#include "bocStructure.h"
#include "rodConfiguration.h"
#include "router.h"

#if (defined(I_AM_MASTER_DSP) || defined(I_AM_SLAVE_DSP))
	#include "primFuncts.h"
#endif

/*************************************************************************************
 *                                     GENERAL
 *************************************************************************************
 *
 * Primitive lists sent from one CPU to another and reply lists returned from the
 * recipient CPU have a 3 parts:
 *      list header:  tagged by the structure MSG_LIST_HEAD declared in msgBuff.h
 *      list body:    composed of the list of primitive or reply messages
 *      list trailer: tagged by the structure MSG_LIST_TAIL declared in msgBuff.h
 *
 * The primitive and reply messages in the list body have 2 parts:
 *      message header: tagged by the structure MSG_HEAD declared in msgBuff.h
 *      message body:   (optional) primitive specific data.  If there is primitive
 *                     specific data, there should be a structure tag for it defined
 *                     in primParams.h (this file).  There should be separate tags for
 *                     input and/or output data as needed.
 *
 * The primitive functions themselves never see the listHeader, listBody or msgHeader.
 * These structures are interpreted by the interface routines, readListWrapper() and
 * execPrim() in listManager.c.  Because of this, changes in the common structures do
 * not require changes to any of the primitive functions, only to one or the other of
 * the interface routines.  This reduces effort but more importantly reduces the
 * possibility of errors.
 *
 * On receipt of a new primitive list, readListWrapper() extracts the information from
 * the list header and the list trailer and checks it for errors.  This information
 * guides the execution of the list.
 *
 * execPrim() is called once for each primitive in a primitive list.  execPrim()
 * extracts the information from the message header, checks it for errors and branches
 * to the appropriate primitive function.  Primitive functions are all of type INT32
 * and have a single argument, a pointer to a structure of type PRIM_DATA declared in
 * primFuncts.h.  execPrim() fills the PRIM_DATA structure before branching to the
 * primitive function.
 *
 * All primitive functions receive a pointer to struct PRIM_DATA as an argument.
 * There will be many authors of primitives, so it is important that the use of this
 * structure is clear.  The structure is described in detail here, followed by
 * instructions of how to add a primitive, with an example.
 *
 *************************************************************************************
 *                          The PRIM_DATA data structure
 *************************************************************************************
 *
 * PRIM_DATA is a 5 element structure which is passed to each primitive function. It
 * is the sole argument passed to the primitive functions and is filled for all
 * primitives by the function execPrim() in listManager.c.  It's use by a primitive
 * function may be tricky so requires care!  In particular, 2 of the elements in
 * PRIM_DATA are themselves pointers to structures defined by the author of each
 * primitive.  These are priBodyPtr and repBodyPtr which point to data used by the
 * primitive and returned from the primitive, respectively.  The structures tags which
 * describe this data are defined by the primitive authors in this file (primParams.h).
 * The elements of PRIM_DATA are now described:
 *
 * UINT32 *priBodyPtr - This is a pointer to the (optional) message body.  As
 * described above, the message body contains primitive specific data which should be
 * described by a structure whose tag is in this file (primParams.h).  The executable
 * on the CPU which sent the primitive list is also built with primParams.h and so
 * uses the same structures to build the primitives.  This ensures that the data
 * described by the structure in primParams.h is what the primitive function receives.
 *
 * UINT32 priBodyLength - This is the number of 4 byte words in the primitive body.
 * This may not always be the same for a given primitive if, for example, one of the
 * primitive data structure elements is of variable length, as in the example below.
 *
 * UINT32 *repBodyPtr - This is a pointer to the (optional) message body of the reply
 * data.  If there is reply data from the primitive, this pointer tells the primitive
 * function where it should put that data.  Again, the author of the primitive is
 * responsible to define a structure tag to describe this data.  That tag should also
 * be in this file (primParams.h).
 *
 * UINT32 repBodyLength - This is the number of 4 byte words in the reply message body.
 * It should NOT include the reply message header, tagged by MSG_HEAD, which is added
 * by execPrim() and which the primitive function knows nothing about.  execPrim()
 * initializes repBodyLength to 0; the primitive function does not need to set it to 0
 * if there is no reply data.
 *
 * UINT32 *repBuffEnd - This points to the last valid address where a primitive can
 * write reply data.  The primitive function can use this to check that the bounds of
 * the reply buffer are not exceeded.  The error code MSG_EXCEEDS_LIST_BOUND can be
 * returned if the reply message has been truncated.  The interface routine, execPrim(),
 * also checks that a primitive has not not exceeded the boundaries of the reply buffer.
 *
 * The interface routines do not look at the input data, they just pass a pointer to
 * the data. So just as the primitive routines are not affected by changes in the
 * common structures, the interface routines are not affected by changes to primitive
 * functions and the associated data structures.  This division means that only people
 * familiar with a particular piece of code ever have to edit it, and it limits the
 * amount of code which is affected by changes.  This all reduces the likelihood of
 * introducing bugs when making such changes.
 *
 *************************************************************************************
 *          readListWrapper(), execPrim() and CPU to CPU consistency checks
 *************************************************************************************
 *
 * Each primitive has up to five items defined in this header file:
 *
 *      ID                            (required)
 *      revision number               (required)
 *      structure tag for input data  (required if there is input data)
 *      structure tag for output data (required if there is reply data)
 *      legal values of parameters    (recommended if there are illegal values)
 *
 * Primitive functions are called from the interface function, execPrim(), based on
 * the primitive ID, which is an element of the MSG_HEAD structure.  The executables
 * on both the sending and receiving CPUs are built with this header file.  The CPU
 * which builds and sends the primitive loads ID into MSG_HEAD to communicate which
 * primitives are to be executed.  The CPU which receives the primitive list extracts
 * ID from MSG_HEAD and branches to a primitive functions based on ID.
 *
 * The branching is done by execPrim().  Within execPrim() the primitive ID is used
 * as an index of an array of structures which has one array element per primitive.
 * The structure tag, PRIM_PARAMETERS, is defined in primFuncts.h.  It has 2 elements:
 *
 * struct PRIM_PARAMETERS {
 *    INT32 (*primFunction)(struct PRIM_DATA *);
 *    INT32 primRevision;
 * };
 *
 * primFunction is a pointer to the primitive function.  primRevision is the current
 * revision of the primitive.  This is needed to ensure the executables of the sending
 * and receiving CPUs were built with the same version.
 *
 * Because primitive IDs index an array of pointers to primitive functions, they
 * must be sequential.  Defining the IDs according to the model below, e.g.
 *
 * #define ECHO              0
 * #define MEMORY_TEST       (1 + (ECHO))
 * #define SET_ERRMSG_MASK   (1 + (MEMORY_TEST))
 *
 * etc. ensures that the primitive IDs are sequential.
 *
 * Note: The primitive IDs have been divided into 3 sets: common to both master & slave
 * DSPs, slave only, master only. New primitives belonging to a given set should be
 * placed at the end of the set, so that the previously written primitive lists
 * stored on the host computers still remain valid. They are still ordered sequentially
 * within the sets to simplify the listManager code. (Table lookups in execPrim,
 * checkPrimHeader and initializePrimParams translate the IDs into indices).
 *
 * The revision number is intended to ensure that both CPUs, sender and receiver,
 * are using the same version of a primitive.  In particular, there could be problems
 * if the definition of the input data structure or the output data structure is not
 * the same on both CPUs.  The primitive revision is an element, primRevision, of the
 * MSG_HEAD structure which is filled by the sending CPU.  The receiving CPU checks
 * the transmitted version of primRevision in the message header with its local copy
 * in the PRIM_PARAMETERS structure and aborts with an error message if there is a
 * mismatch. This check is performed in a single place, in the interface routine
 * execPrim() in listManager.c.  It should NOT be performed by the primitive
 * functions.  In fact, execPrim() does not pass the revision number to the primitive.
 *
 * There are no restrictions on the mnemonic constants which are used to define the
 * ID, the revision and the structure tags for the input and output data of the
 * primitive functions.  However, because these constants will be used not only by the
 * author of the primitive but also by the authors of code which builds primitive
 * lists, there should be some consistency in how they are defined.  The following
 * model has been used so far:
 *    The ID should indicate what the primitive does.
 *    The revision should be the ID preceded by R_ and should be of the form xxyy
 *   where xx is the major revision number and yy is the minor revision number.
 *    The input structure tag should be the ID appended with _IN
 *    The output structure tag should be the ID appended with _OUT.
 * For example say we have primitive which sorts the input data and returns this as
 * output data and that this primitive follows a primitive with ID ECHO in the list.
 * We would have:
 * #define SORT_DATA    (1 + (ECHO))            ID is mnemonic and sequential
 * #define R_SORT_DATA           300            revision number 3.00
 *    struct SORT_DATA_IN {                     structure tag for input data
 *      char *unsorted;
 *    };
 *    struct SORT_DATA_OUT {                    structure tag for output data
 *      char *sorted;
 *    };
 *
 * In some cases, it may also be useful to define mnemonics for legal values of
 * parameters which are passed in the structures.  See, e.g., SET_LED below.
 *
 * Finally, the constant PRIM_LIST_REVISION, below, is intended to ensure that the
 * primitives which are defined on both CPUs are the same and in the same order, i.e.
 * a given ID number corresponds to the same primitive on both sides.
 * PRIM_LIST_REVISION should be incremented in the following situations:
 *
 * 1) A primitive is added to this file.
 * 2) A primitive is removed from this file.
 * 3) The primitives in this file are reordered.
 * *) It does NOT need to be incremented in the event that individual primitives are
 *    revised. PRIM_REVISION takes care of that.
 *
 * The primitive list and the reply list have a common header structure which is tagged
 * by MSG_LIST_HEAD in msgBuff.h.  This structure has an element, primListRevision,
 * which the sending CPU fills with its copy of PRIM_LIST_REVISION.  The receiving CPU
 * checks this against its copy of PRIM_LIST_REVISION and aborts if there is a
 * mismatch.  This check is performed in a single place, in the interface routine
 * readListWrapper() in listManager.c.
 *
 *************************************************************************************
 *          primitive function skeletons with and without return data
 *************************************************************************************
 *
 * -- SIZEOF(x) is defined as sizeof(x) >> 2, i.e. it is the size in 32 bit words.
 * -- In some cases there might not be a structure for the return data, e.g. if it
 * -- it's format or size is not completely specified.  In those cases, the last
 * -- line would be primData->repBodyLength = N; where N is the size in 32 bit words.
 *    see, e.g., rwFifo in masterPrimFuncts.c.
 *
 * INT32 noReturnDataSkel(struct PRIM_DATA *primData) {
 *
 *   INT32 returnCode = SUCCESS;
 *
 *   struct NO_RETURN_DATA_SKEL_IN *noReturnDataSkelIn =
 *                     (struct NO_RETURN_DATA_SKEL_IN *)primData->priBodyPtr;
 *
 *   -- code goes here --
 *
 *   return returnCode;
 * }
 *
 ***
 *
 * INT32 returnDataSkel(struct PRIM_DATA *primData) {
 *
 *   INT32 returnCode = SUCCESS;
 *
 *   struct RETURN_DATA_SKEL_IN *returnDataSkelIn =
 *                     (struct RETURN_DATA_SKEL_IN *)primData->priBodyPtr;
 *   struct RETURN_DATA_SKEL_OUT *returnDataSkelOut =
 *                     (struct RETURN_DATA_SKEL_OUT *)primData->repBodyPtr;
 *
 *   -- code goes here --
 *
 *   primData->repBodyLength = SIZEOF(struct RETURN_DATA_SKEL_OUT);
 *
 *   return returnCode;
 * }
 *
 * Note that the last line of code which assigns repBodyLength may take other forms.
 * For example the echo primitive (in primFuncts.c) has in the body an arbitrary
 * number of data words given by priBodyLength and pointed to by priBodyPtr both
 * members of the PRIM_DATA structure.  The return message will have the same number
 * of words in its body and they will be placed at repBodyPtr, also a member of the
 * PRIM_DATA structure.  So there we do not have a structure indicating where the
 * primitive and reply data are or their sizes.  repBodyLength is assigned by the line
 *   primData->repBodyLength = primData->priBodyLength.
 * Other examples of this type of exception are in the rwFifo() in masterPrimFuncts.c
 * and in the sortData example below.
 *
 * Note also that in noReturnDataSkel, it was not necessary to set repBodyLength to
 * 0.  This is done in the interface routine, execPrim(), from which all primitives
 * are called.
 *
 *
 *
 *************************************************************************************
 *                                  EXAMPLES
 *                   How to add a primitive in 5 easy steps.
 *************************************************************************************
 *
 * Here, the procedure to add a primitive function, checkBit(), is described.
 * checkBit() checks whether a bit in a register is set. The result is returned in a
 * reply message. checkBit is a primitive common to both types of DSPs, so it is
 * placed at the end of the common section.
 *
 *  1) In primParams.h, define the constant mnemonic for the primitive ID, the
 *     revision number and structure tags for the input and output (reply) data.
 *
 *    #define CHECK_BIT (1 + (<previously last primitive>))
 *    #define R_CHECK_BIT   100
 *       struct CHECK_BIT_IN {
 *         UINT32 registerId;
 *         UINT32 bitNumber;
 *       };
 *       struct CHECK_BIT_OUT {
 *         UINT32 bitSet;
 *       };
 *
 *    #define LAST_COMMON_PRIMITIVE         (CHECK_BIT)
 *
 *  2) In primParams.h, near the bottom, add the prototype for the primitive function.
 *
 *    INT32 checkBit(struct PRIM_DATA *);
 *
 *  3) In primParams.h increment PRIM_LIST_REVISION.
 *
 *  4) Update primIDList & the primParameters array inside initializePrimParams in
 *     primFuncts.c; add in the new primitive in the appropriate spot (bottom
 *     of common, slave-only, or master-only functions):
 *
 *     primIDList[++primLoop]=                  CHECK_BIT;
 *     primParameters[primLoop].primFunction=   checkBit;
 *     primParameters[primLoop].primRevision=   R_CHECK_BIT;
 *
 *  5) In primFuncts.c, slavePrimFuncts.c or masterPrimFuncts.c, write the routine...
 *
 *    INT32 checkBit(struct PRIM_DATA *primData) {
 *
 *      INT32 returnCode = SUCCESS;
 *      INT32 error;
 *      UINT32 registerId;
 *      UINT32 bitNumber;
 *      UINT32 bitSet;
 *
 *      struct CHECK_BIT_IN *checkBitIn =
 *                        (struct CHECK_BIT_IN *)primData->priBodyPtr;
 *      struct CHECK_BIT_OUT *checkBitOut =
 *                        (struct CHECK_BIT_OUT *)primData->repBodyPtr;
 *
 *      registerId = checkBitIn->registerId;
 *      bitNumber = checkBitIn->bitNumber;
 *
 *      error = readRegister( registerId, 1, bitNumber, &bitSet);
 *
 *      if (error < 0) {
 *        addError(&returnCode,error, "checkBit","readRegister", __FILE__, __LINE__);
 *        if (FATAL(returnCode)) { return returnCode; }
 *      }
 *
 *      checkBitOut->bitSet = bitSet;
 *      primData->repBodyLength = SIZEOF(struct CHECK_BIT_OUT);
 *
 *      return returnCode;
 *    }
 *
 *  (Note how an error return from a function call is handled.)
 *
 * Now we add a primitive function, sortData(), just after checkBit(). sortData sorts
 * input byte data and returns the sorted list. Like checkBit, sortData has both input
 * and output data. However, neither the input nor the output data has an associated
 * structure.  They are after all just unstructured data lists of unspecified length.
 *
 *  1) In primParams.h, define the constant mnemonic for the primitive ID and the
 *    revision number. In this case there are no structure tags.
 *
 *    #define SORT_DATA (1 + (CHECK_BIT))
 *    #define R_SORT_DATA   100
 *
 *    #define LAST_COMMON_PRIMITIVE         (SORT_DATA)
 *
 *  2) In primParams.h, near the bottom, add the prototype for the primitive function.
 *
 *    INT32 sortData(struct PRIM_DATA *);
 *
 *  3) In primParams.h increment PRIM_LIST_REVISION.
 *
 *  4) Update primIDList & the primParameters array inside initializePrimParams:
 *
 *     primIDList[++primLoop]=                  SORT_DATA;
 *     primParameters[primLoop].primFunction=   sortData;
 *     primParameters[primLoop].primRevision=   R_SORT_DATA;
 *
 *  5) In primFuncts.c, slavePrimFuncts.c or masterPrimFuncts.c, write the routine...
 *
 *    INT32 sortData(struct PRIM_DATA *primData) {
 *
 *      INT32 returnCode = SUCCESS;
 *      char *inPtr, *outPtr;
 *      char tmp;
 *      UINT32 numElements, outer, inner;
 *
 *      inPtr  = primData->priBodyPtr;
 *      outPtr  = primData->repBodyPtr;
 *      numElements = 4 * primData->priBodyLength;
 *
 *      if ((UINT32)inPtr + numElements > (UINT32)primData->repBuffEnd) {
 *        newError(&returnCode, MSG_EXCEEDS_LIST_BOUND, FATAL_ERR, "sortData",
 *                 "reply data will exceed reply buffer boundary...aborting",
 *                 __FILE__, __LINE__);
 *        return returnCode;
 *      }
 *
 *      for (outer = 0; outer < (numElements - 2); ++outer) {
 *        for (inner = 0; inner < (numElements - 2 - outer); ++inner) {
 *          if ( *(inPtr + (char *)inner) > *(inPtr + (char *)(inner+1))) {
 *            tmp = *(inPtr + (char *)inner);
 *            *(inPtr + (char *)inner) = *(inPtr + (char *)(inner+1));
 *            *(inPtr + (char *)(inner+1)) = tmp;
 *          }
 *        }
 *        *(outPtr + (char *)(numElements-1-outer)) =
 *                      *(inPtr + (char *)(numElements-1-outer));
 *      }
 *      *outPtr = *inPtr;
 *
 *      primData->repBodyLength = primData->priBodyLength;
 *
 *      return returnCode;
 *    }
 *
 * Note that here a check has been added to make sure the reply data does not exceed
 * the reply data buffer.
 *
 *                                                        D Fasching
 */
#define PRIM_LIST_REVISION     111





/************************************************************************************
 *                        Primitives common to all DSPs go here
 * Note: some of these primitives (eventTrapSetup, triggerMask, setTrigger) are used
 * to coordinate the DSPs with each other or other electronics on the ROD (such as
 * the router). These primitives should be sent to the master DSP from the host; the
 * master DSP primitive will then set the appropriate board registers, etc. and pass
 * on the primitive to the slave DSP. The primitive passing software will not break
 * if the host sends one of these primitives directly to a slave, but slave
 * applications might break as a result of inconsistent parameter settings.
 ************************************************************************************/
#define COMMON_PRIMITIVES_BASE 0x0

/* Echo the primitive message body back in the reply message body. */
#define ECHO                    (COMMON_PRIMITIVES_BASE)
#define R_ECHO                  100
/* echo's input & output data is unstructured. */

/*
 * Set error severity mask - levels which are masked will not write an error message
 * to the text buffer.
 *  IN: errMsgMask: value to set errMsgMask to;
 *                  DEFAULT means do not change the value
 *  OUT: errMsgMaskOut->errMsgMask: value of errMsgMask at end of this routine
 */
#define SET_ERRMSG_MASK         (1 + (ECHO))
#define R_SET_ERRMSG_MASK       100
	struct SET_ERRMSG_MASK_IN {
		UINT32 errMsgMask;
	};
	struct SET_ERRMSG_MASK_OUT {
		UINT32 errMsgMask;
	};

/*
 * Pause the execution of the primitive list.  List execution will resume when the
 * RESUME bit is set in the command register.
 */
#define PAUSE_LIST              (1 + (SET_ERRMSG_MASK))
#define R_PAUSE_LIST            100

/* eventTrapSetup:
 * Sets the event trapping parameters for a particular slave.  They need to be set
 * in the slave DSP and in the router fpga. Common to Master/Slave, however
 * eventTrapSetup is intended to be a HOST to MASTER DSP primitive only; the master
 * DSP then sets up the router registers and passes a primitive list to call
 * eventTrapSetup on the slave DSP.  Trap configuation is not independent, if only
 * one trap is used, it must be trap 0. The primary purpose for trap 1 is to have
 * a way of running short immediate tasks with a different function (like trapping
 * events) on the DSP, while it continues a task such as occupancy counting.
 */
#define EVENT_TRAP_SETUP        (1 + (PAUSE_LIST))
#define R_EVENT_TRAP_SETUP      103
	struct EVENT_TRAP_SETUP_IN {
		UINT32 slvBits, numberOfEvents, timeoutInUsec;
		UINT32 extRouterSetup, distribute;
		UINT32 releaseFrames, permitBackPressure, dataMode, sLink,
		       format, trapStray,  iterLimit;
		UINT32 trapConfig[2], trapExclusionFlag[2], trapFunction[2],
		       trapMatch[2],  trapModulus[2],       trapRemainder[2];
	};
	struct EVENT_TRAP_SETUP_OUT {
		UINT32 errorCode;
	};
	#define  COLLECT_FOREVER  0
	#define  KEEP_EVENTS      0

	#define  TRAP_DISTRIB_PRIMARY    0
	#define  TRAP_DISTRIB_SECONDARY  1
	#define  TRAP_DISTRIB_BOTH       2
	#define  TRAP_DISTRIB_NEITHER    3

	#define  TRAP_FMT_NORMAL  0
	#define  TRAP_FMT_ERROR   1

/* Quickly set a section of the DSP's memory */
#define SET_MEMORY           (1 + (EVENT_TRAP_SETUP))
#define R_SET_MEMORY         100
	struct SET_MEMORY_IN {
		UINT32 *start, size, val;
	};

/* Quickly copy one section of the DSP's memory into another section */
#define COPY_MEMORY          (1 + (SET_MEMORY))
#define R_COPY_MEMORY        100
	struct COPY_MEMORY_IN {
		UINT32 *source, *destination, size;
	};

/* New memory test, designed to work with full code (ie not the memory-test mode) */
#define MEMORY_TEST          (1 + (COPY_MEMORY))
#define R_MEMORY_TEST        102
	struct MEMORY_TEST_IN {
		UINT32 *start, size, repetitions[6], errorsBeforeFail, continueOnError;
		UINT32 nReads, dmaFlag;
	};
	struct MEMORY_TEST_OUT {
		UINT32 returnCode;
	};

/* Turn an LED on or off. */
#define SET_LED              (1 + (MEMORY_TEST))
#define R_SET_LED            104
	struct SET_LED_IN {
		UINT32  ledNum, ledState;
	};
	#define YELLOW_LED 0
	#define GREEN_LED  1
	#define RED_LED    2

	#define OFF     0
	#define ON      1
	#define TOGGLE  2

/* Flash an LED.  Period and number of flashes are settable parameters. */
#define FLASH_LED            (1 + (SET_LED))
#define R_FLASH_LED          103
	struct FLASH_LED_IN {
		UINT32 ledNum, period, numTimes;
	};

#define SEND_DATA            (1 + (FLASH_LED))
#define R_SEND_DATA          103
	struct SEND_DATA_IN {
		UINT32 dataType, auxVal, repBufferFlag, timeout;
	};
	struct SEND_DATA_OUT {
		void   *dataPtr;
		UINT32 dataLength;
	};
	#define MIRROR_DATA    0x10
	#define ROUTER_DATA    0x11
	#define STREAM_DATA    0x12

	#define EVENT_DATA     0x20
	#define HISTOGRAM_DATA 0x21
	#define OCCUPANCY_DATA 0x22
	#define FIT_DATA       0x23
	#define BIN_DATA       0x24

/* moduleMask defines the link setup (command & data) for the modules. Like
   eventTrapSetup, moduleMask is meant to be sent from the host to the master DSP,
   which will transfer its data to the slave DSP(s) if desired. */
#define MODULE_MASK         (1 + (SEND_DATA))
	#define R_MODULE_MASK       101

	#if defined(I_AM_SLAVE_DSP)
		struct MODULE_MASK_IN {
			ModuleMaskData moduleMaskData[N_TOTMODULES];
		};
	#else
		struct MODULE_MASK_IN {
			UINT32 moduleNum, port, useStructSet, passOn, slvBits;
			UINT32 cmdLine, dataLine[4];
			UINT32 cfg, modMask[2], maskType, storage, maskSet; 
		};
	#endif

/* Send a trigger to the detector electronics. If setTrigger is issued to a
   slave DSP, the trigger source is assumed as external (possibly not even issued
   by the master DSP); in this case only the current bin & histogram set are set. */
#define SET_TRIGGER          (1 + (MODULE_MASK))
#define R_SET_TRIGGER        102
	struct SET_TRIGGER_IN {
		struct  CmdList cmdList[2];
		UINT32  slvBits;
		UINT16  cmdBuff, preBuilt;
		UINT32  chipAddress, bin, set;
		UINT32  repetitions, interval;
		UINT32  preBuiltDataOffset[2];
		UINT16  incCmd[2];
		UINT32  incData[2];

	};

/* Serial port definitions; used by several primitives & tasks: */
#define SP0      0
#define SP1      1
#define SP_BOTH  2

#define START_TASK           (1 + (SET_TRIGGER))
#define R_START_TASK         101
	/* Task revision numbers must be updated whenever a task's input parameters
	   are modified. Revision numbers of the tasks should not exceed 0xff. If
	   a task has a default position for its memory base (eg. trapTask), that is
	   defined inside memoryPartitions.h along with the other memory-related
	   constants. */

	/* master */
/* Task which controls an entire histogram scan */
	#define R_HISTOGRAM_CTRL_TASK   107

		struct XPair {
			FLOAT32 x0, delta_x;
		};
		struct RangeList {
			struct XPair xPair[5]; 
		};

	#if defined(SCT_ROD)

	struct  HISTOGRAM_CTRL_TASK_IN {
		UINT8   slvBits, port, configRegister[2];

		UINT8   configSctSet, dataSet;
		UINT8   groupRangeMap[2];

		UINT8   groupDSPMap[4];

		UINT8   groupSPMap[2];
		UINT8   globalCtrl, syncLevel;

		UINT32  *histoBase;

		UINT8   doChipOcc, dataFormat, binSize, unused1;
		UINT8   extSetup, dataPath, capture, useRangeList /* unused2 */;

		UINT32  repetitions, nBins, bin0;

		struct  RangeList rangeList[2];
		FLOAT32 *dataPtr[2];

		struct  CmdList triggerSequence[2];
		UINT8   incCmd[2];
		UINT8   calLineLoop;
		UINT8   distributionToggle;

		UINT32  incData[2];

		UINT32  genData[10];
	};

	#elif defined(PIXEL_ROD)

		struct  HISTOGRAM_CTRL_TASK_IN {
			ScanControl scanControl;
		};
	#endif

	#define MODULE_BASIC  0
	#define MODULE_TRIM   1
	#define MODULE_ALL    2

	#define EXT_SETUP_NONE    0
	#define EXT_SETUP_RCF     1 
	#define EXT_SETUP_ROUTER  2
	#define EXT_SETUP_SET     4
	#define EXT_SETUP_HISTO   8 
	#define EXT_SETUP_HTASK   16 

	#define DATA_PATH_NORMAL  0
	#define DATA_PATH_INMEM   1

	#define MODULE_GROUP   0
	#define ROUTER_DISTRIB 1

	#define SYNC_NEWBIN    0
	#define SYNC_TOTAL     1

	struct HISTOGRAM_CTRL_TASK_OUT {
		UINT32  totalTime;
		UINT32  slvProcTime[4];
		UINT32  *dataPtr, dataLength;
	};

	#define R_MIRROR_TASK     101
	struct MIRROR_TASK_IN {
		UINT32 slvBits, mirrorFreq, *mirrorBase,
		       *mirrorSlvBase, mirrorLen;
	};
	#define MIRROR_DEFAULT_BASE 0x800000a0

	#define R_TRAP_REQ_TASK   101
	struct TRAP_REQ_TASK_IN {
		UINT32 slvBits, watchFreq;
	};

	/* slave */
	#define R_HISTOGRAM_TASK  100
	struct HISTOGRAM_TASK_IN {
		UINT32 nEvents, controlFlag;  /* nEvents can equal COLLECT_FOREVER too. */
	};
	/* control flag settings: */
	#define MASTER_HREG     0
	#define LOCAL_INC       1
	#define LOCAL_SET_TRIG  2

	struct HISTOGRAM_TASK_OUT {
		UINT32  nEvents, binsDone;
		FLOAT32 avgProcTime;
		UINT32  *dataPtr, dataLength;
	};

	#define R_TRAP_TASK       100
	struct TRAP_TASK_IN {
		UINT32 nEvents, reloadInterval, trapType, eventType,
		       *trapBufferBase, trapBufferLength;
	};

	struct TRAP_TASK_OUT {
		UINT32  nEvents, dataLen;
		UINT32  *bufferBase, bufferLen;
	};

	#define R_OCCUPANCY_TASK  100
	struct OCCUPANCY_TASK_IN {
		UINT32 nEvents, nFilters, splitFlag;
        char   filter[16];
	};

	#define R_ERROR_TASK      100
	struct ERROR_TASK_IN {
		UINT32 errorType;
	};
	struct ERROR_TASK_OUT {
		UINT32 nErrors;
	};

	#define R_RESYNCH_TASK    100
	struct RESYNCH_TASK_IN {
		UINT32 errorType;
	};

	/* tasks can have their own specialized output structures, this is a
	   general one. It is stored as an array of external structures in
	   the task manager, so that if a task completes & later the data is
	   requested, it can be supplied. */
	struct GEN_TASK_OUT {
		UINT32 *dataPtr, dataLength;
	};
	union TASK_STRUCTURES_OUT {
		struct HISTOGRAM_CTRL_TASK_OUT  histoCtrlTaskOut;
		struct HISTOGRAM_TASK_OUT       histogramTaskOut;
		struct TRAP_TASK_OUT            trapTaskOut;
		struct ERROR_TASK_OUT           errorTaskOut;
		struct GEN_TASK_OUT             genTaskOut;
	};

	/* MAX_NUM_TASKS is the maximum of the two task numbers for master &
	   slave DSPs. Bad Things can happen if it is not kept up to date */
	#define MAX_NUM_TASKS   5

	/* The IDs of the two task sets cannot not exceed 0xff, and must increment by 1. */
	/* master DSP tasks: */
	#define MASTER_TASK_BASE     (0x10)
	#define HISTOGRAM_CTRL_TASK  (MASTER_TASK_BASE)
	#define MIRROR_TASK          (1+(HISTOGRAM_CTRL_TASK))
	#define TRAP_REQ_TASK        (1+(MIRROR_TASK))

	#define LAST_MASTER_TASK  (TRAP_REQ_TASK)
	#define NUM_MASTER_TASKS  ((LAST_MASTER_TASK)-(MASTER_TASK_BASE)+1)
			#define ALL_SLAVES      5

	/* slave DSP tasks: */
	#define SLAVE_TASK_BASE  (0x20)
	#define HISTOGRAM_TASK  (SLAVE_TASK_BASE)
	#define TRAP_TASK       (1+(HISTOGRAM_TASK))
	#define OCCUPANCY_TASK  (1+(TRAP_TASK))
	#define ERROR_TASK      (1+(OCCUPANCY_TASK))
	#define RESYNCH_TASK    (1+(ERROR_TASK))

	#define LAST_SLAVE_TASK  (RESYNCH_TASK)
	#define NUM_SLAVE_TASKS  ((LAST_SLAVE_TASK)-(SLAVE_TASK_BASE)+1)

	#if   defined(I_AM_MASTER_DSP)
		#define LAST_TASK  (LAST_MASTER_TASK)
		#define NUM_TASKS  (NUM_MASTER_TASKS)
		#define TASK_BASE  (MASTER_TASK_BASE)
	#elif defined(I_AM_SLAVE_DSP)
		#define LAST_TASK  (LAST_SLAVE_TASK)
		#define NUM_TASKS  (NUM_SLAVE_TASKS)
		#define TASK_BASE  (SLAVE_TASK_BASE)
	#endif

	union TASK_STRUCTURES_IN {
		struct HISTOGRAM_CTRL_TASK_IN  histoCtrlTaskIn;
		struct MIRROR_TASK_IN          mirrorMemoryTaskIn;
		struct TRAP_REQ_TASK_IN        trapRequestTaskIn;

		struct HISTOGRAM_TASK_IN  histogramTaskIn;
		struct TRAP_TASK_IN       trapTaskIn;
		struct OCCUPANCY_TASK_IN  occupancyTaskIn;
		struct ERROR_TASK_IN      errorTaskIn;
		struct RESYNCH_TASK_IN    resynchTaskIn;
	};

	struct START_TASK_IN {
		UINT32 taskType, taskRevision, priority, completionFlag;
		union  TASK_STRUCTURES_IN  taskStruct;
	};

#define TASK_OPERATION      (1 + (START_TASK))
#define R_TASK_OPERATION         100
	struct TASK_OPERATION_IN {
		UINT32 taskType, taskOperation, data;
	};

#define TASK_STOP         0
#define TASK_PAUSE        1
#define TASK_RESUME       2
#define TASK_QUERY        3
#define TASK_RESET        4
#define TASK_SETPRIORITY  5

#define TEST         (1 + (TASK_OPERATION))
#define R_TEST              100
	struct TEST_IN {
		UINT32 dataLen, *dataPtr;
	};

#define WRITE_BUFFER         (1 + (TEST))
#define R_WRITE_BUFFER         101
	struct WRITE_BUFFER_IN {
		UINT32 buffer;
		/*char string[64];*/
		char string[2396];  /* sizeof(union PRIM_PARAMS_UNION) => 2400 bytes. */

	};
	#define WRITE_BUFFER_ERR  0
	#define WRITE_BUFFER_INFO 1
	#define WRITE_BUFFER_DIAG 2
	#define WRITE_BUFFER_XFER 3

#define LAST_COMMON_PRIMITIVE   (WRITE_BUFFER)
#define NUM_COMMON_PRIMITIVES   ((LAST_COMMON_PRIMITIVE)-(COMMON_PRIMITIVES_BASE)+1)

#if   defined(SCT_ROD)
	#define COMMON_SCT_PRIMITIVES_BASE 0x800

	#ifdef COMMENTED
	#define XXX (COMMON_SCT_PRIMITIVES_BASE)
	#define LAST_COMMON_SCT_PRIMITIVE   (XXX)
	#define NUM_COMMON_SCT_PRIMITIVES   \
		((LAST_COMMON_SCT_PRIMITIVE)-(COMMON_SCT_PRIMITIVES_BASE)+1)
	#endif

	#define NUM_COMMON_SCT_PRIMITIVES  0

#elif defined(PIXEL_ROD)
	#define COMMON_PIXEL_PRIMITIVES_BASE 0x800

	#ifdef COMMENTED
	#define XXX (COMMON_PIXEL_PRIMITIVES_BASE)
	#define LAST_COMMON_PIXEL_PRIMITIVE   (XXX)
	#define NUM_COMMON_PIXEL_PRIMITIVES   \
		((LAST_COMMON_PIXEL_PRIMITIVE)-(COMMON_PIXEL_PRIMITIVES_BASE)+1)
	#endif

	#define NUM_COMMON_PIXEL_PRIMITIVES  0

#endif

/************************************************************************************
 *                             Slave DSP primitives go here.
 ************************************************************************************/
#define SLAVE_PRIMITIVES_BASE  0x1000

#define START_EVENT_TRAPPING   (SLAVE_PRIMITIVES_BASE)
#define R_START_EVENT_TRAPPING       101

#define STOP_EVENT_TRAPPING    (1 + (START_EVENT_TRAPPING))
#define R_STOP_EVENT_TRAPPING        100

/* Set up & define the histograms. */
#define HISTOGRAM_SETUP        (1 + (STOP_EVENT_TRAPPING))
#define R_HISTOGRAM_SETUP            108
	struct HISTOGRAM_SETUP_IN {
		UINT32  *base;
		UINT32  nBins;
		UINT32  binSize;
		UINT32  routineType;
		UINT32  dataType[2];
		UINT32  opt[4];

		UINT32  validModules[2];
		UINT32  moduleRangeMap[2][2];
		MDAT32 *xPtr[2];
	};
	struct HISTOGRAM_SETUP_OUT {
		UINT32 *base, size;
		UINT32 *ctrlPtr, *binCtr, *varRangePtr;
	};

	/* The default histogram base is defined inside memoryPartitions.h along with
	   the other memory-related constants. */
	#define HISTOGRAM_CONDENSED 1
	#define HISTOGRAM_FULL      3

	#define HISTOGRAM_8BIT      8
	#define HISTOGRAM_16BIT     16
	#define HISTOGRAM_32BIT     32
	
	#define HISTO_ROUTINE_C     0
	#define HISTO_ROUTINE_ASM   1


#define LAST_SLAVE_PRIMITIVE    (HISTOGRAM_SETUP)
#define NUM_SLAVE_PRIMITIVES   ((LAST_SLAVE_PRIMITIVE)-(SLAVE_PRIMITIVES_BASE)+1)

#if   defined(SCT_ROD)
	#define SLAVE_SCT_PRIMITIVES_BASE 0x1800

	#ifdef COMMENTED
	#define XXX (SLAVE_SCT_PRIMITIVES_BASE)
	#define LAST_SLAVE_SCT_PRIMITIVE   (XXX)
	#define NUM_SLAVE_SCT_PRIMITIVES   \
		((LAST_SLAVE_SCT_PRIMITIVE)-(SLAVE_SCT_PRIMITIVES_BASE)+1)
	#endif

	#define NUM_SLAVE_SCT_PRIMITIVES  0

#elif defined(PIXEL_ROD)
	#define SLAVE_PIXEL_PRIMITIVES_BASE 0x1800

	#ifdef COMMENTED
	#define XXX (SLAVE_PIXEL_PRIMITIVES_BASE)
	#define LAST_SLAVE_PIXEL_PRIMITIVE   (XXX)
	#define NUM_SLAVE_PIXEL_PRIMITIVES   \
		((LAST_SLAVE_PIXEL_PRIMITIVE)-(SLAVE_PIXEL_PRIMITIVES_BASE)+1)
	#endif

	#define NUM_SLAVE_PIXEL_PRIMITIVES  0

#endif

/************************************************************************************
 *                          Master DSP primitives go here.
 ************************************************************************************/
#define MASTER_PRIMITIVES_BASE 0x2000

/* Read or write a block of slave memory.
 * - slaveNumber: ID of slave to be accessed
 * - readNotWrite = 0 for writing to slaves, != 0 for reading from slaves
 * - *slaveAddress: slave reads: source address of the data
 *                  slave writes: destination address of the data
 * - *masterAddress: slave reads: destination address of the data; the special value
 *                                DEFAULT puts the data in the body of the reply msg
 *                                just after the RW_SLAVE_MEMORY_OUT structure
 *                   slave writes: source address of the data; the special value
 *                                DEFAULT gets the data from the body of the primitive
 *                                just after the RW_SLAVE_MEMORY_IN structure
 * - numWords: in input structure: number of 32 bit words to read or write */
#define RW_SLAVE_MEMORY         (MASTER_PRIMITIVES_BASE)
#define R_RW_SLAVE_MEMORY       100
	struct RW_SLAVE_MEMORY_IN {
		UINT32 slaveNumber;
		UINT32 readNotWrite;
		UINT32 *slaveAddress;   /* DEFAULT address defined above */
		UINT32 *masterAddress;  /* DEFAULT address defined above */
		UINT32 numWords;
	};

/* Transmit serial data to controller FPGA */
#define TRANS_SERIAL_DATA (1+(RW_SLAVE_MEMORY))
#define R_TRANS_SERIAL_DATA 100
	struct TRANS_DATA {
		UINT32 port0Data; /* 32b serial data port 0 */
		UINT32 port1Data; /* 32b serial data port 1 */
	};
 	struct TRANS_SERIAL_DATA_IN {
		UINT32   captureSerOn;      /* Capture module return data in input FIFOs */
		UINT32   streamLen[2], *streams;   /* data sets to transmit */
	};

/* Set DSPINT to start slave. */
#define START_SLAVE_EXECUTING     (1 + (TRANS_SERIAL_DATA))
#define R_START_SLAVE_EXECUTING   104
	struct START_SLAVE_EXECUTING_IN {
		UINT32 slaveNumber;
		UINT32 commOnOff;
		UINT32 slaveType;
		UINT32 timeoutInUsec;
	};
	struct START_SLAVE_EXECUTING_OUT {
		UINT32 slaveNumber;
	};
	/* commOnOff options */
	#define SLV_DSP_COMM_OFF      0
	#define SLV_DSP_COMM_ON       1
	/* slaveType options */
	#define SLV_DSP_UNCONFIGURED  0
	#define SLAVE_CALIBRATE       1
	#define SLAVE_MONITOR         2
	#define SLAVE_ERROR_CHECK     3
	#define SLAVE_MEMORY_TEST     4

/* Sets the parameters of the structure rodConfig.slvDspConfig[slvId]. */
#define CONFIG_SLAVE            (1 + (START_SLAVE_EXECUTING))
#define R_CONFIG_SLAVE          102
	struct CONFIG_SLAVE_IN {
		UINT32 slaveNumber;
		UINT32 commOnOff;
		UINT32 slaveType;
	};  /* parameters values are same as for START_SLAVE_EXECUTING */

/*
 *  Reads or writes a field of a register in the master DSP memory map.
 * registerID: ID of the register being polled
 * offset: number of lsb in the field, starting with 0
 * width: width of the field
 *    mnemonic symbols for registerID, offset and width are in registerIndices.h
 * readNotWrite: 1 for read, 0 for write
 * dataIn: data to write for readNotWrite = 0; no significance for readNotWrite = 1
 * dataOut: data read for readNotWrite = 1; no significance for readNotWrite = 0
 */
#define RW_REG_FIELD                 (1 + (CONFIG_SLAVE))
#define R_RW_REG_FIELD               105
	struct RW_REG_FIELD_IN {
		UINT32 registerID;
		UINT32 offset;
		UINT32 width;
		UINT32 readNotWrite;
		UINT32 dataIn;
	};
	struct RW_REG_FIELD_OUT {
		UINT32 dataOut;
	};

/*
 *  Polls a field of a register for a value.  A timeout is available.
 * registerID: ID of the register being polled
 * offset: number of lsb in the field, starting with 0
 * width: width of the field
 *    mnemonic symbols for registerID, offset and width are in registerIndices.h
 * desiredValue: value of the defined field to poll for
 * timeoutInUsec: timeout period in microseconds
 * found: indicates whether 'desiredValue' was found within 'timeoutInUsec'
 */
#define POLL_REG_FIELD               (1 + (RW_REG_FIELD))
#define R_POLL_REG_FIELD             105
	struct POLL_REG_FIELD_IN {
		UINT32 registerID;
		UINT32 offset;
		UINT32 width;
		UINT32 desiredValue;
		UINT32 timeoutInUsec;
	};
	struct POLL_REG_FIELD_OUT {
		UINT32 found;
	};

/* Reads/writes some number of words from/to one of the FIFOs on the ROD.
 * - fifoID: INPUT_MEM, DEBUG_MEM or EVENT_MEM
 * - bank: BANK_A, BANK_B or (EVENT_MEM only) BANK_C
 * - readNotWrite: 1 for read, 0 for write
 * - numElements: number of fifo entries to read or write; NOTE: no packing, eg
 *              to write 20 elements to INPUT_MEM BANK_A requires 40 32b data words
 *              to write 20 elements to EVENT_MEM BANK_C requires 20 32b data words
 *              20 elements read from EVENT_MEM BANK_C will occupy 20 32b data words
 * - *dataBaseAdr: for readNotWrite = 0 source address of data for the FIFO
 *                                = 1 destination address for data from the FIFO
 *   if *dataBaseAdr = DEFAULT
 *     readNotWrite = 0: data is supplied with the RW_FIFO primitive after *data
 *     readNotWrite = 1: data is written to the reply message body after bytesXfrd
 *   There is also a scratch space in SDRAM at SCRATCH_BASE (memoryPartitions.h)
 * - bytesXfrd: Number of bytes read or written
 */
#define RW_FIFO                   (1 + (POLL_REG_FIELD))
#define R_RW_FIFO                 104
	struct RW_FIFO_IN {
		UINT32 fifoId;
		UINT32 bank;
		UINT32 readNotWrite;
		UINT32 numElements;
		UINT32 *dataBaseAdr;    /* DEFAULT value defined above */
	};
	struct RW_FIFO_OUT {
		UINT32 bytesXfrd;
	};
	#define INPUT_MEM 0x0             /* values of fifoID */
	#define DEBUG_MEM 0x1
	#define EVENT_MEM 0x2
	#define TIM_MEM   0x3
	#define BANK_A    0x0             /* values of bank; ignored for TIM_MEM */
	#define BANK_B    0x1
	#define BANK_C    0x2             /* EVENT_MEM only */

/*
 *  Send a primitive list to a slave DSP.
 * slaveNumber: ID of slave DSP to recieve the list.
 * listLength: number of 32 bit words in the list.
 * *slavePrimList: Address of start of source data for the list.
 *   - *slavePrimList = DEFAULT indicates that the slave primitive list is stored in
 *     the body of this SEND_SLAVE_LIST primtive starting just after the last
 *     SEND_SLAVE_LIST_IN structure element.
 * *slaveRepData: Address where master should put slave reply data if the
 *             START_SLAVE_LIST primitive is executed with getSlaveReply != 0.
 *   - *slaveRepData = DEFAULT indicates that the slave reply list should be stored
 *     in the body of a reply message to the START_SLAVE_LIST primtive.
 */
#define SEND_SLAVE_LIST      (1 + (RW_FIFO))
#define R_SEND_SLAVE_LIST    103
	struct SEND_SLAVE_LIST_IN {
		UINT32 slaveNumber;
		UINT32 listLength;
		UINT32 *slavePrimList;    /* DEFAULT value defined above */
		UINT32 *slaveRepData;     /* DEFAULT value defined above */
	};

/*
 *  Start a slave DSP executing its primitive list by setting its inListRdy bit.
 * slaveNumber: ID of slave DSP to execute list
 * pauseMasterList != 0 if master DSP list execution pauses while slave executes list
 * getSlaveReply != 0 if the master should read the slave reply data
 */
#define START_SLAVE_LIST      (1 + (SEND_SLAVE_LIST))
#define R_START_SLAVE_LIST    103
struct START_SLAVE_LIST_IN {
	UINT32 slaveNumber;
	UINT32 pauseMasterList;
	UINT32 getSlaveReply;
};

/* pause, resume or abort a slave list */
#define SLAVE_LIST_OP      (1 + (START_SLAVE_LIST))
#define R_SLAVE_LIST_OP    101
	struct SLAVE_LIST_OP_IN {
		UINT32 slaveNumber;
		UINT32 listOp;
	};
	#define LIST_PAUSE  0
	#define LIST_RESUME 1
	#define LIST_ABORT  2

/* Build & send a command stream to the detector electronics. sendData will ship
   a command stream back to the VME host. */
#define BUILD_STREAM          (1 + (SLAVE_LIST_OP))
#define R_BUILD_STREAM        102
	struct BUILD_STREAM_IN {
		struct  CmdList cmdList;
		UINT32  cmdBuff, reset, chip, fibre;
		UINT32  dataLen;    /* Length of the extra data, in bits. */
		UINT32  *data;      /* Start of the data for a command with large amounts of
		                       data (masks & some pixel commands). */
	};

#define SEND_STREAM          (1 + (BUILD_STREAM))
#define R_SEND_STREAM        100
	struct SEND_STREAM_IN {
		UINT32 cmdBuff, captureSerOn;
	};

/* configure a module, or send it's data (resident in memory) to control links.
   For input data, the pointer to configData is the start. */
#define RW_MODULE_DATA     (1 + (SEND_STREAM))
#define R_RW_MODULE_DATA   102
	struct RW_MODULE_DATA_IN {
		UINT32 readNotWrite, structId, moduleNum;
		Module *configData;
	};
	struct RW_MODULE_DATA_OUT {
		Module configData;
	};
	#define PHYSICS_MODULE_CONFIG  0
	#define SCAN_MODULE_CONFIG     1
	#define SPARE_MODULE_CONFIG    2
	#define N_MODULE_CONFIG_SETS ((SPARE_MODULE_CONFIG) -(PHYSICS_MODULE_CONFIG) +1)


#define SEND_CONFIG      (1 + (RW_MODULE_DATA))
#define R_SEND_CONFIG    105
	struct SEND_CONFIG_IN {
		UINT32   port,  captureSerOn, moduleNum[2], chipNum,
		         setLinks, restore, structId, groupId, dataType,
		         activeOnly, enableDataTaking;
	};

	#define NO_CONFIG_LOOP        0
	#define NORMAL_CONFIG_LOOP    1
	#define OFF_ROD_CONFIG_LOOP   2

	#if defined(SCT_ROD)
	#define CONFIG_MODULE_BASIC  0
	#define CONFIG_MODULE_TRIM   1
	#define CONFIG_MODULE_ALL    2

	#elif defined(PIXEL_ROD)
	//Pixel additive bitwise definitions; the 1st 14 correspond to the bit
	//definitions in the pixel control register.
	#define CONFIG_MODULE_ENABLE       0
	#define CONFIG_MODULE_SELECT       1
	#define CONFIG_MODULE_PREAMP       2
	#define CONFIG_MODULE_HITBUS       3

	#define CONFIG_MODULE_TDAC_0       4
	#define CONFIG_MODULE_TDAC_1       5
	#define CONFIG_MODULE_TDAC_2       6
	#define CONFIG_MODULE_TDAC_3       7
	#define CONFIG_MODULE_TDAC_4       8

	#define CONFIG_MODULE_FDAC_0       9
	#define CONFIG_MODULE_FDAC_1       10
	#define CONFIG_MODULE_FDAC_2       11
	#define CONFIG_MODULE_FDAC_3       12
	#define CONFIG_MODULE_FDAC_4       13

	#define CONFIG_MODULE_GLOBAL       16
	#define CONFIG_MODULE_CONTROL      17
	#define CONFIG_MODULE_TDAC         18
	#define CONFIG_MODULE_FDAC         19

	#define CONFIG_MODULE_SCAN_TDAC    20
	#define CONFIG_MODULE_SCAN_FDAC    21
	#define CONFIG_MODULE_MCC          22
	#define CONFIG_MODULE_ALL          23
	#endif


	#define ALL_MODULES 0xfe
	#define NO_MODULE   0xff

	#define MODULE_GROUP_0       0
	#define MODULE_GROUP_1       1
	#define MODULE_GROUP_2       2
	#define MODULE_GROUP_3       3
	#define MODULE_GROUP_4       4
	#define MODULE_GROUP_5       5
	#define MODULE_GROUP_6       6
	#define MODULE_GROUP_7       7

	#define MODULE_GROUP_ALL     8
	#define N_MODULE_GROUPS      (MODULE_GROUP_ALL)

#define DSP_RESET      (1 + (SEND_CONFIG))
#define R_DSP_RESET    100
	struct DSP_RESET_IN {
		UINT32  slvBits, forceSync, nAttempts, timeOut;
	};

#define SET_ROD_MODE   (1 + (DSP_RESET))
#define R_SET_ROD_MODE    101
	struct SET_ROD_MODE_IN {
		UINT32  mode, flag, fifoSetup, nBits, delay, evtsPerL1A, message;
	};

#define RW_MODULE_VARIABLE  (1 + (SET_ROD_MODE))
#define R_RW_MODULE_VARIABLE    102
	struct RW_MODULE_VARIABLE_IN {
		UINT32  read,  structId, groupId, module, chip, 
		        varType;

		UINT32  info;     /* for writing: output message? */
		UINT32 dataLen;   /* for writing: length of data (e.g. mask= 4) */
		MDAT32 *data;     /* for writing: *data is written to all selected chips/modules */
	};
	struct RW_MODULE_VARIABLE_OUT {
		UINT32  nModData, dataLen;
		MDAT32 *data;
	};

	#define MVAR_GROUP_ID    100
	#define MVAR_ACTIVE      101
	#define MVAR_SELECT      102


#define RW_BOC_DATA  (1 + (RW_MODULE_VARIABLE))
#define R_RW_BOC_DATA    100
	struct RW_BOC_DATA_IN {
		UINT32  read, sendToBoc, dataLen, *data;
	};
	struct RW_BOC_DATA_OUT {
		BOCConfig bocCfgData;
	};


#define BOC_HISTOGRAM  (1 + (RW_BOC_DATA))
#define R_BOC_HISTOGRAM    100
	struct BOC_HISTOGRAM_IN {
		UINT32  numSamples, numLoops;
	};
	struct BOC_HISTOGRAM_OUT {
		UINT32 histo[96][2];
	};

#define LAST_MASTER_PRIMITIVE   (BOC_HISTOGRAM)
#define NUM_MASTER_PRIMITIVES   ((LAST_MASTER_PRIMITIVE)-(MASTER_PRIMITIVES_BASE)+1)


#if   defined(SCT_ROD)
	#define MASTER_SCT_PRIMITIVES_BASE 0x2800

	#ifdef COMMENTED
	#define XXX (MASTER_SCT_PRIMITIVES_BASE)
	#define LAST_MASTER_SCT_PRIMITIVE   (XXX)
	#define NUM_MASTER_SCT_PRIMITIVES   \
		((LAST_MASTER_SCT_PRIMITIVE)-(MASTER_SCT_PRIMITIVES_BASE)+1)
	#endif

	#define NUM_MASTER_SCT_PRIMITIVES  0

#elif defined(PIXEL_ROD)
	#define MASTER_PIXEL_PRIMITIVES_BASE 0x2800

	#define TEST_GLOBAL_REG (MASTER_PIXEL_PRIMITIVES_BASE)
	#define R_TEST_GLOBAL_REG   112
		struct TEST_GLOBAL_REG_IN {
			UINT32 structId;
			UINT32 moduleId;
			UINT32 FEIndex; 
		};
		struct TEST_GLOBAL_REG_OUT {
			UINT32 testResult;
			UINT32 dataLen;
			MDAT32 *data;
		};

	#define LAST_MASTER_PIXEL_PRIMITIVE   (TEST_GLOBAL_REG)
	#define NUM_MASTER_PIXEL_PRIMITIVES   \
		((LAST_MASTER_PIXEL_PRIMITIVE)-(MASTER_PIXEL_PRIMITIVES_BASE)+1)
#endif


#if   defined(SCT_ROD)
	#define NUM_PRIMITIVES ( (NUM_COMMON_PRIMITIVES)     \
	                        +(NUM_SLAVE_PRIMITIVES)      \
	                        +(NUM_MASTER_PRIMITIVES)     \
	                        +(NUM_COMMON_SCT_PRIMITIVES) \
	                        +(NUM_SLAVE_SCT_PRIMITIVES)  \
	                        +(NUM_MASTER_SCT_PRIMITIVES)   )

#elif defined(PIXEL_ROD)
	#define NUM_PRIMITIVES ( (NUM_COMMON_PRIMITIVES)       \
	                        +(NUM_SLAVE_PRIMITIVES)        \
	                        +(NUM_MASTER_PRIMITIVES)       \
	                        +(NUM_COMMON_PIXEL_PRIMITIVES) \
	                        +(NUM_SLAVE_PIXEL_PRIMITIVES)  \
	                        +(NUM_MASTER_PIXEL_PRIMITIVES)   )

#else
	#define NUM_PRIMITIVES ( (NUM_COMMON_PRIMITIVES) \
	                        +(NUM_SLAVE_PRIMITIVES)  \
	                        +(NUM_MASTER_PRIMITIVES)   )

#endif

/************************************************************************************
 *  Primitive function prototypes (not needed by the host processor).
 ************************************************************************************/
#if (defined(I_AM_MASTER_DSP) || defined(I_AM_SLAVE_DSP))
INT32 noPrimitive(struct PRIM_DATA *);

INT32 echo(struct PRIM_DATA *);
INT32 setErrMsgMask(struct PRIM_DATA *);
INT32 pauseList(struct PRIM_DATA *);
INT32 eventTrapSetup(struct PRIM_DATA *);
INT32 setMemory(struct PRIM_DATA *);
INT32 copyMemory(struct PRIM_DATA *);
INT32 memoryTest(struct PRIM_DATA *);
INT32 setLed(struct PRIM_DATA *);
INT32 flashLed(struct PRIM_DATA *);
INT32 sendData(struct PRIM_DATA *);
INT32 moduleMask(struct PRIM_DATA *);
INT32 setTrigger(struct PRIM_DATA *);
INT32 startTask(struct PRIM_DATA *);
INT32 taskOperation(struct PRIM_DATA *);
INT32 test(struct PRIM_DATA *);
INT32 writeBuffer(struct PRIM_DATA *);

#if   defined(SCT_ROD)
#elif defined(PIXEL_ROD)
#endif

INT32 startEventTrapping(struct PRIM_DATA *);
INT32 stopEventTrapping(struct PRIM_DATA *);
INT32 histogramSetup(struct PRIM_DATA *);

#if   defined(SCT_ROD)
#elif defined(PIXEL_ROD)
#endif

INT32 rwSlaveMemory(struct PRIM_DATA *);
INT32 transSerialData (struct PRIM_DATA *);
INT32 startSlaveExecuting(struct PRIM_DATA *);
INT32 configSlave(struct PRIM_DATA *);
INT32 rwRegField(struct PRIM_DATA *);
INT32 pollRegField(struct PRIM_DATA *);
INT32 rwFifo(struct PRIM_DATA *);
INT32 sendSlaveList(struct PRIM_DATA *);
INT32 startSlaveList(struct PRIM_DATA *);
INT32 slaveListOp(struct PRIM_DATA *);
INT32 buildStream(struct PRIM_DATA *);
INT32 sendStream(struct PRIM_DATA *);
INT32 rwModuleData(struct PRIM_DATA *);
INT32 sendConfig(struct PRIM_DATA *);
INT32 dspReset(struct PRIM_DATA *);
INT32 setRodMode(struct PRIM_DATA *);
INT32 rwModuleVariable(struct PRIM_DATA *);
INT32 rwBocData(struct PRIM_DATA *);
INT32 BOChistogram(struct PRIM_DATA *primData);
/* INT32 configBoc(struct PRIM_DATA *); */

#if   defined(SCT_ROD)
#elif defined(PIXEL_ROD)
	INT32 testGlobalReg(struct PRIM_DATA *);  
#endif

#endif  /* primitive parameters definition block */

#endif
