/*
 * DefaultInterpreter.java
 *
 * Created on 28 November 2002, 13:56
 */

package GuiComponents.scripting;
import GuiComponents.Console.*;
import org.apache.bsf.*;
import java.io.*;
import java.awt.event.*;
import java.util.*;

/**
 * @author Matthew Palmer
 *
 * This class is the top level interface to the BSF
 * It has a console and does some standard things like reading from the streams
 * For various reasons, specific languages may want a modified version of this class, 
 * they should sub-class it.
 *
 * It would use the GUIConsole interface from bsh except the interface doesn't have half the methods of
 * JConsole!  So we just use JConsole directly for now.  This is less likely to need sub-classing, but you never know!
 *
 * Most of what this class does is sit in a loop trying to read from the console input stream.
 * It also provides some useful functions that it processes rather than passing them to the underlying scripting engine
 */
public class DefaultInterpreter implements ActionListener, NameCompletionListener, Interpreter {
    protected JConsole console; 
    protected BSFEngine engine;
    protected NameCompleter completer;
 
    protected String defaultPath;
    protected String consoleCommandString = ".";
    protected char endCommandChar = '\n';
    protected char lineContinueChar = '\\';
    protected char blockStartChar = '{';
    protected char blockEndChar = '}';
    
    private int nestLevel = 0;
    private StringBuffer buffered = new StringBuffer();
    
    /** Creates a new instance of DefaultInterpreter */
    public DefaultInterpreter(JConsole console, BSFEngine engine) {
        this(console,  engine, null);
    }
    
    public DefaultInterpreter(JConsole console, BSFEngine engine, NameCompleter name) {
        this.completer = name;
        this.console = console;
        console.addActionListener(this);
        console.addNameCompletionListener(this);
        console.setPrompt("bsf> ");
        this.engine = engine;
        printStartMaterial();
        console.initialize();
        defaultPath = System.getProperty("sct.daq.root") + java.io.File.pathSeparatorChar + "share" +java.io.File.pathSeparatorChar + "scripts" + java.io.File.pathSeparatorChar ;
    }
    
    public JConsole getConsole() {
	return console;
    }
    
    /**
     * Should be overridden to return false when executing a line that returns void.
     * Note, should always return false if the return value is null
     */
    protected boolean printObject(Object val) {
        if (val == null) return false;
        return true;
    }
    
    /**
     * Returns true if this line starts a block
     */
    protected boolean startBlock(String line) {        
        StringBuffer sb = new StringBuffer(line.trim());
        if ((sb.length() > 0) && (sb.charAt(sb.length()-1) == blockStartChar)) return true;
        return false;
    }
    
    /**
     * Return true if this line ends a block
     */
    protected boolean endBlock(String line) {        
        StringBuffer sb = new StringBuffer(line.trim());
        if ((sb.length() > 0) && (sb.charAt(sb.length()-1) == blockEndChar)) return true;
        return false;
    }
    
    /**
     * Default behaviour is to see if line ends with a line continuation character
     */
    protected boolean continueLine(String line) {        
        StringBuffer sb = new StringBuffer(line.trim());
        if ((sb.length() > 0) && (sb.charAt(sb.length()-1) == lineContinueChar)) return true;
        return false;
    }
    
    public String toString() {
        return "DefaultInterpreter";
    }
      
    protected int nestLevel() {
        return nestLevel;
    }
    
        
    /**
     * Prints an indent - helpful for the user when typing in blocks
     */
    protected void incIndent() {
        nestLevel++;
        setIndent(nestLevel);
    }
    
    protected void decIndent() {        
        nestLevel--;
        if (nestLevel < 0) nestLevel = 0;            
        setIndent(nestLevel);
    }
    
    protected void setIndent(int level) {
        String indent="";
        for (int i=0; i<level; i++)
            indent += "\t";
        console.setPostPrompt(indent);
    }
    
    /**
     * Prints the initial blurb you see
     * Sub-classes should override to tell the user how to get the help text at least
     */
    protected void printStartMaterial() {        
    }
    
    /**
     * Prints some help for the user
     */
    protected void printHelp() {
        
    }
    
    /**
     * Executes a console command 
     * The string should include the initial char
     */
    protected void consoleCommand(String command)  {
        if (command.startsWith(".x")) 
            loadFile(command.substring(2).trim());
            
    }
    
    /**
     * Load the contents of file and execute them.
     */
    protected void loadFile(String file) {
        File f = new File(file);
        if (!f.exists() || !f.isFile()) {
            if (file.indexOf(java.io.File.pathSeparatorChar) == -1) {
                loadFile(defaultPath+file);
            } else {
                console.println("Not a valid filename: " + file);
            }
            return;
        }
        
        //Read the whole file.
        try {
            FileReader fr = new FileReader(f);
            char[] buffer = new char[(int)f.length()];      //Can't read long files!
            fr.read(buffer);
            
            //Read, so execute it in the interpreter
            exec(new String(buffer), file, 0, 0);
        
        } catch (IOException ioe) {
            console.println("Error opening file: " + ioe.toString());
        }               
    }
    
    /** Invoked when we need to complete a name.
     * Do some minimal filtering and pass it onto the NameCompleter objects.
     */
    public List completeName(String curCmd) {
        if (completer == null) return null;
        
        //Find last occurrence of a space after triming
        curCmd = curCmd.trim();
        int index = curCmd.lastIndexOf(" ");
        String context;
        String part;
        if (index != -1) {
            context = curCmd.substring(0, index);
            part = curCmd.substring(index + 1);
        } else {
            context = "";
            part = curCmd;
        }
                
        List l = completer.completeName(context, part);	
	//System.out.println("Completions");
	for (int i=0; i<l.size(); ++i) {
	    l.set(i, context + (String)l.get(i));
	    //System.out.println(l.get(i));
	}
	//System.out.println();
	return l;
    }
    
    
    /**
     * Executes the string.
     */
    public void exec(String s, String source, int row, int column) {
        try {            
            Object retVal = engine.eval(source, row, column, s);
            //engine.exec(source, row, column, s);
            if (printObject(retVal))
                console.println(retVal.toString());
	    else 
		console.println("[null]");
        } catch (BSFException be) {
            console.println(be.toString());
            //System.out.println(be.getMessage());
            //be.printStackTrace();
        }            
    }
    
    public void exec(String s) {        
        exec(s, this.toString(), -1, -1);
    }
    
    /** Invoked when an action occurs.
     *
     */
    public void actionPerformed(ActionEvent e) {
        String line = e.getActionCommand();        
        
        if (line.startsWith(consoleCommandString)) {
            consoleCommand(line);     
            return;
        }
            
        if (continueLine(line)) {
            buffered.append(line);
            return;
        }

        if (startBlock(line)) {
            buffered.append(line);
            incIndent();
            return;
        }

        if (endBlock(line)) {
            decIndent();
        }

        if (nestLevel > 0) {
            buffered.append(line);
            return;
        }

        line = buffered + line;
        buffered.delete(0, buffered.length());      //Empty buffer
        
        exec(line);                                 //Execute!
    }        
}
