#ifndef NAGFITSTRATEGY_H
#define NAGFITSTRATEGY_H
#include "../FitStrategy.h"
#include <nag.h>
#include <nage04.h>
#include <vector>

using namespace Sct;
using namespace std;

namespace SctFitter{
    /**
       Concrete fit strategy uses the NAG C function 
       <A HREF="http://wwwasd.web.cern.ch/wwwasd/lhc++/Nag_C/CL/doc/e/e04fcc.html">e04fcc</A> 
       "Unconstrained nonlinear least squares (no derivatives required)"
	<P>
	At the time of writing, CERN licences are available to CERN collaborators from the ANAPHE site:
	http://anaphe.web.cern.ch/anaphe/download2.html
	<P>
	To use NagFitStrategy, define NAGDIR, and rebuild.
       @author Alan Barr
       @date 16/5/03
     */

    class NagFitStrategy : public FitStrategy {
    public:
	/// constuctor
	NagFitStrategy(string opt=string("")) throw();
	/// destructor
	~NagFitStrategy() throw();
	/// fit TH1
	virtual void fitTH1(const TH1& hist, TF1& fit) const throw(LogicError, MathsError);
	/// fit TGraph @note errors set to unity
	virtual void fitTGraph(const TGraph& graph, TF1& fit) const throw(LogicError, MathsError);
	/// fit TGraphErrors
	virtual void fitTGraphErrors(const TGraphErrors& graph, TF1& fit) const throw(LogicError, MathsError);
	
	/// Returns the name of the strategy
	virtual const string& getName() const throw();
    private:
	NagFitStrategy( const NagFitStrategy& ); ///< no copy constructor
	const string name; ///< the name of the strategy
	static bool inMap; ///< dummy variable used at static initialization.
	
	/**
	   utility routine @return the number of points to fit
	   @return number of active bins 
	   @param histogram (from which bin positions used)
	   @param fit function (from which limits found)
	   @param returns with the first bin from which to fit
	   @note if options does not contain 'R' whole histogram is used.
	*/
       	const int getNumberFitPoints(TH1& hist, const TF1& fit, 
					    int& iFirstBin) const throw(LogicError);
	/**
	   utility routine finds active points in graph
	   @param graph the graph of points
	   @param fit the fit from which the range is taken
	   @param the vector of bools saying which points in the graph to fit.
	   @note if options does not contain 'R' whole histogram is used.	   
	 */

	void getFitPoints(TGraph& graph, const TF1& fit, vector<bool>& active) const throw(LogicError);
	/// Interal class caches the information needed for the fit.
	class Cache{
	public:
	    /// constructor extracts all of the relevant information from hist.
	    Cache(const TH1& hist, const unsigned npoints, 
		  const unsigned firstpoint, TF1& fit) throw(LogicError, MathsError) ;

	    /// constructor extracts information from the TGraph;
	    Cache(const TGraph& graph, TF1& fit, const vector<bool>& active) throw(LogicError, MathsError) ;

	    /// constructor extracts information from the TGraphErrors;
	    Cache(const TGraphErrors& graph, TF1& fit, const vector<bool>& active) throw(LogicError, MathsError) ;

	    ///Sets up pars to contain all the correct parameters
	    void convertPars(double inPars[]);
	    	    
	    ///Sets up the arrays pars and map
	    void setupPars(TF1& fit); 
	    
	    ~Cache() throw();                ///< destructor
	    vector<double> m_y; ///< histogram contents.
	    vector<double> m_x; ///< histogram bin centers.
	    vector<double> m_ey; ///< histogram error.
	    TF1* function;           ///< pointer to the function.
	    double* pars;		///< holds the internal array of parameters - used when some are fixed
	    unsigned int nVarPars;	///<Number of variable params
	    double* inPars;		///<Used to hold the params that are varied by Nag
	    
	private:	    
	    int* map;		        ///<Maps input pars to output pars
	};

	/**
	   contains the NAG calls etc - the guts of the fit.
	*/
	void nagFit(Cache* cache, TF1& fit, bool quiet) const throw(MathsError);

	/// Take range from TF1? @return true if options contain 'R' or 'r'
	bool ranged() const throw();

	/// Fit in quiet mode? @return true if options contain 'Q' or 'q'
	bool quiet() const throw();

	/// calculates chi-squared from cache.
	static void chiSquaredFunctionForNag(Integer m, Integer n, double x[], double fvec[],
					     Nag_Comm *comm) throw();
    };

} // end of namespace SctFitter;


#endif // #ifndef NAGFITSTRATEGY_H
