#include "StrobeDelayFitAlgorithm.h"
#include "FitAlgorithmMap.h"
#include "SctData/FitScanResult.h"
#include "SctData/StrobeDelayVariable.h"
#include "SctData/StandardDefects.h"
#include "SctData/TopHatFitObject.h"
#include <TH1.h>

using namespace SctData;

namespace SctFitter {
    
bool StrobeDelayFitAlgorithm::inMap = FitAlgorithmMap::instance().setAlgorithm(StrobeDelayVariable::instance().getVariableName(), auto_ptr<FitAlgorithm>(new StrobeDelayFitAlgorithm));

StrobeDelayFitAlgorithm::StrobeDelayFitAlgorithm() throw() {
    FitterMode mode = getMode();
    mode.fitNone(); 
    mode.fitChips(); 
    mode.doSummary(false); 
    setMode(mode);
}

auto_ptr<FitObject> StrobeDelayFitAlgorithm::getPrototype() const throw() {
    return auto_ptr<FitObject> (new TopHatFitObject());
}

void StrobeDelayFitAlgorithm::guessParameters(const TH1& hist, FitObject& fitOb) const throw (LogicError, MathsError) {
    fitOb.setParameter(0, hist.GetMaximum() ) ;
    // mid-rise and rise-width
    fitOb.setParameter(1, findLevel(hist,0.5,true)) ;                 // 50% level
    double p2 = findLevel(hist, 0.82, true) - fitOb.getParameter(1);    // level 1-sigma away from 50%
    if (p2<0.) p2=-p2;
    fitOb.setParameter(2, p2 ) ;
    
    // mid-fall and fall-width
    fitOb.setParameter(3, findLevel(hist,0.5,false)) ;                 // 50% level
    double p4 = findLevel(hist, 0.82, false) - fitOb.getParameter(3);    // level 1-sigma away from 50%
    if (p4<0.) p4=-p4;
    fitOb.setParameter(4, p4 ) ;
    
    int lbin=findBin(hist, 0.01, true); if ( (lbin-1) >= 1 ) lbin--;
    int ubin=findBin(hist, 0.01, false); if ( (ubin+1) <= hist.GetNbinsX() ) ubin++;
    fitOb.setVarMin(0 , hist.GetBinLowEdge( lbin ) );
    fitOb.setVarMax(0 , hist.GetBinLowEdge( ubin + 1 ) );    
}

void StrobeDelayFitAlgorithm::checkForDefects(const FitObject& fo, const ModuleElement& element,
						DefectList& defects) const {
  if (fo.getNDF() &&  fo.getChiSquared()/fo.getNDF() > StandardDefects::BADFIT.getParameter() ) {
      cout << "Chi-squared: " << fo.getChiSquared() << " NDF: " << fo.getNDF() << " so claiming is DEAD" << endl;
      defects.addDefect(Defect(StandardDefects::BADFIT,element));
  }
}

void StrobeDelayFitAlgorithm::checkForDefects(const TH1& hist, const ModuleElement& element, 
					      DefectList& defects) const throw (LogicError) {
    double maxVal = hist.GetMaximum();
    double minVal = hist.GetMinimum();
    
    if ( maxVal < StandardDefects::DEAD.getParameter() ){
	defects.addDefect(Defect(StandardDefects::DEAD, element));
    } 
    if( minVal > StandardDefects::STUCKON.getParameter() ){
	defects.addDefect(Defect(StandardDefects::STUCKON, element));
    } 
    if( maxVal < StandardDefects::UNDER.getParameter() ){
	defects.addDefect(Defect(StandardDefects::UNDER,element));
    }else if( maxVal > StandardDefects::OVER.getParameter() ){
	defects.addDefect(Defect(StandardDefects::OVER,element));
    }
}

void StrobeDelayFitAlgorithm::createSummaryHistograms(FitScanResult& fits) const throw() {
}    
    
}
