#include "Test.h"
#include "Sct/SctNames.h"
#include "Sct/IoExceptions.h"
#include "Sct/LogicErrors.h"
#include <sstream>
#include <boost/date_time/posix_time/posix_time.hpp>

using namespace std;
using namespace Sct;
using namespace boost::posix_time;

namespace SctCalibrationController {

Test::Test(const TestData& d, const list<string>& l) {
    //Now check consistency
    if (d.testPoints_size != d.nScans) 
	throw InvalidArgumentError("testPoints size not the same as the number of Scans in Test:" + getUniqueID(), __FILE__, __LINE__);
    
    data = d;
    data.testPoints = new double[data.testPoints_size];
    for (unsigned int i=0; i<data.testPoints_size; ++i) {
	data.testPoints[i] = d.testPoints[i];
    }
    string startTime = to_iso_string(second_clock::universal_time());
    data.startTime = startTime.substr(0, 15);
    
    data.modules = new string[l.size()];
    data.modules_size = l.size();
    unsigned int count = 0;
    for (list<string>::const_iterator i=l.begin(); i!=l.end(); ++i, ++count) {
        data.modules[count] = *i;
    }


    
    publish();
}

Test::~Test() {
    withdraw();
}

void Test::addScan(Sct_SctApi_T_Scan* scan) {
    if (scans.size() == data.nScans)
	throw InvalidArgumentError("Tried to add too many Scans to Test: " + getUniqueID(), __FILE__, __LINE__);
    scans.push_back(scan);
}

const TestData& Test::getData() const {
    return data;
}

string Test::getUniqueID() const {
    ostringstream s;
    s << "TestData." << data.runNumber << "." << data.startScanNumber;
    return s.str();    
}

Sct_SctApi_T_Scan* Test::getScan(unsigned int index) const {
    if (index >= scans.size()) {
	ostringstream s;
	s << "Tried to getScan with index: " << index << " but only " << scans.size() << " available in Test: " << getUniqueID();
	throw InvalidArgumentError(s.str(), __FILE__, __LINE__);
    }
    return scans[index];
}

list<string> Test::getModuleList() const{
    list<string> list;
    for (unsigned int i=0; i<data.modules_size; ++i) {
        list.push_back(data.modules[i]);
    }
    return list;
}

void Test::setStatus(TestData::status_E status) {
    //Check first
    if (data.status == TestData::ABORTED && status != TestData::ABORTED) {
	throw InvalidArgumentError("Can't change status from ABORTED in Test: " + getUniqueID(), __FILE__, __LINE__);
    }
    if (data.status == TestData::EXECUTING && status == TestData::COMPLETED && scans.size() != data.nScans) {
	throw InvalidArgumentError("Can't enter COMPLETED state when not all Scans have been added in Test: " + getUniqueID(), __FILE__, __LINE__);
    }
    if (data.status == TestData::COMPLETED && status != TestData::COMPLETED) {
	throw InvalidArgumentError("Can't change status from COMPLETED in Test: " + getUniqueID(), __FILE__, __LINE__);
    }
    if (data.status != status && (status == TestData::COMPLETED || status == TestData::ABORTED)) {
	string endTime = to_iso_string(second_clock::universal_time());
	data.endTime = endTime.substr(0, 15);
    }
    
    data.status = status;
    publish();
}
    
void Test::publish() {
    string name = SctNames::getControlDataName();
    name += ".";
    name += getUniqueID();
    ISInfoDictionary& is = SctNames::getISDictionary();
    ISInfo::Status s;
    
    if (is.contains(name.c_str())) {
	s = is.update(name.c_str(), data);
    } else {
	s = is.insert(name.c_str(), data);
    }
    
    if (s != ISInfo::Success) {
	IsException error(s, "Test::publish failed", __FILE__, __LINE__);
	error.sendToMrs(MRS_DIAGNOSTIC);
    }    
}

void Test::withdraw() {
    string name = SctNames::getControlDataName();
    name += ".";
    name += getUniqueID();
    ISInfoDictionary& is = SctNames::getISDictionary();
    ISInfo::Status s;
    
    s = is.remove(name.c_str());
    if (s == ISInfo::CommFailure) {
	IsException error(s, "Test::withdraw failed", __FILE__, __LINE__);
	error.sendToMrs(MRS_DIAGNOSTIC);
    }    
}

}
