#include "PipelineAlgorithm.h"
#include "AnalysisAlgorithmMap.h"
#include "SctData/PipelineTestResult.h"
#include "SctData/StandardDefects.h"
#include "SctData/OccupancyProjector.h"
#include <TH1.h>
#include <TH2.h>

using namespace std;
using namespace boost;
using namespace SctData;
using namespace Sct;

namespace SctAnalysis {
    
    bool PipelineAlgorithm::inMap = AnalysisAlgorithmMap::instance().setAlgorithm("PipelineTest", auto_ptr<AnalysisAlgorithm>(new PipelineAlgorithm()));
    
    shared_ptr<AnalysisAlgorithm> PipelineAlgorithm::clone(const TestData& testData, const string& moduleName) const throw() {
	return shared_ptr<AnalysisAlgorithm>(new PipelineAlgorithm(testData, moduleName, *this));
    }

    shared_ptr<TestResult> PipelineAlgorithm::createTestResult() const {
	return shared_ptr<TestResult>(new PipelineTestResult());
    }
    
    void PipelineAlgorithm::loadData() {
	loadAllRaws();
    }
    
    bool PipelineAlgorithm::canAnalyze() const {
	return PipelineAlgorithm::hasAllRaws();
    }
    
    void PipelineAlgorithm::analyze() {
	shared_ptr<PipelineTestResult> result = dynamic_pointer_cast<PipelineTestResult>(getTestResult());
	result->setScanVariable(getRaw(0)->getHeader().getVariable());	
	shared_ptr<const RawScanResult> rawa=getRaw(0);
	shared_ptr<const RawScanResult> rawb=getRaw(1);
	
	bool pass = true;

	// quick check:
	if (rawa->getScanData(0).GetNbinsY() != rawb->getScanData(0).GetNbinsY() ||
	    rawa->getScanData(1).GetNbinsY() != rawb->getScanData(1).GetNbinsY() ){
	    ostringstream os;
	    os << "PipelineAlgorithm different y-extent of scans " << rawa->getHeader().getScanNumber()
	       << " and " << rawb->getHeader().getScanNumber() <<ends;
	    throw IllegalStateError(os.str(), __FILE__, __LINE__);
	}

	DefectList& defects=result->getDefects();

	/// loop over chips
	for (unsigned ichip=0; ichip<nChipModule; ++ichip){

	    bool stuck[nChannelChip], dead[nChannelChip];

	    // loop to find stuck and dead channels and report stuck and dead cells
	    for (unsigned ichInChip=0; ichInChip<nChannelChip; ++ichInChip){
		// initialise
		stuck[ichInChip]=false;
		dead[ichInChip]=false;

		// find the real channel number
		unsigned ichannel=ichip*nChannelChip+ichInChip;

		TH2& hista = rawa->getScanData(ichip/nChipLink);
		TH2& histb = rawb->getScanData(ichip/nChipLink);
		unsigned nstuckcell=0, ndeadcell=0;

		unsigned nPipeline=hista.GetNbinsY();
		
		// find number of stuck and dead cells for this channel
		for ( unsigned delay=0; delay < nPipeline; ++delay){
		    double vala = hista.GetBinContent(ichannel%nChannelLink+1, delay+1);
		    double valb = histb.GetBinContent(ichannel%nChannelLink+1, delay+1);
		    if(vala<0.5) ++ndeadcell;   // cell is dead val=0
		    if(valb>0.5) ++nstuckcell;  // cell is stuck val >0
		}

		if (nstuckcell==nPipeline){                 // channel stuck
		    stuck[ichInChip]=true;
		} else if (nstuckcell!=0) {                // cell stuck
		    defects.addDefect(Defect(StandardDefects::STUCK_CELL, ModuleElement::Channel(ichannel)));
		    pass = false;
		}
		
		if (ndeadcell==nPipeline){                 // channel dead
		    dead[ichInChip]=true;
		} else  if (ndeadcell!=0) {               // cell dead
		    defects.addDefect(Defect(StandardDefects::DEAD_CELL, ModuleElement::Channel(ichannel)));
		    pass = false;
		}
	    } // end of first loop over channels in chip

	    //loop to report contiguous stuck and dead channels within this chip
	    int firstDeadChannel =-1, lastDeadChannel=-1;
	    int firstStuckChannel=-1, lastStuckChannel=-1;	    
	    for (unsigned ichInChip=0; ichInChip<nChipModule; ++ichInChip){
		// find the real channel number
		unsigned ichannel=ichip*nChannelChip+ichInChip;

		// first dead
		if (dead[ichInChip]) {
		    if (firstDeadChannel==-1) firstDeadChannel=ichannel;
		    lastDeadChannel=ichannel;
		}
		if (!dead[ichInChip] || ichInChip==nChannelChip){
		    if (firstDeadChannel!=-1){
			defects.addDefect(Defect(StandardDefects::DEAD, 
					  ModuleElement(firstDeadChannel, lastDeadChannel)));
			pass = false;
			firstDeadChannel=-1;
			lastDeadChannel=-1;
		    }
		}

		// Now stuck:
		if (stuck[ichInChip]) {
		    if (firstStuckChannel==-1) firstStuckChannel=ichannel;
		    lastStuckChannel=ichannel;
		}
		if (!stuck[ichInChip] || ichInChip==nChannelChip){
		    if (firstStuckChannel!=-1){
			defects.addDefect(Defect(StandardDefects::STUCKON, 
					  ModuleElement(firstStuckChannel, lastStuckChannel)));
			pass = false;
			firstStuckChannel=-1;
			lastStuckChannel=-1;
		    }
		}
	    } // end of second loop over channels in chip
	} // end of loop over chips
	
	result->setPassed(pass);
    }
} // end of namespace SctAnalysis
