#include "FullBypassAlgorithm.h"
#include "AnalysisAlgorithmMap.h"
#include "SctData/FullBypassTestResult.h"
#include "SctData/StandardDefects.h"
#include "SctData/RawScanResult.h"
#include "SctData/OccupancyProjector.h"
#include "TH2.h"
 
using namespace std;
using namespace boost;
using namespace SctData;
using namespace Sct;

namespace SctAnalysis {
    
    bool FullBypassAlgorithm::inMap = AnalysisAlgorithmMap::instance().setAlgorithm("FullBypassTest", auto_ptr<AnalysisAlgorithm>(new FullBypassAlgorithm()));
    
    shared_ptr<AnalysisAlgorithm> FullBypassAlgorithm::clone(const TestData& testData, const string& moduleName) const throw() {
	return shared_ptr<AnalysisAlgorithm>(new FullBypassAlgorithm(testData, moduleName, *this));
    }

    float FullBypassAlgorithm::s_vddNominal=4.0;

    FullBypassAlgorithm::FullBypassAlgorithm() { 
    }
    
    void FullBypassAlgorithm::loadData() {
	loadAllRaws();
    }
    
    bool FullBypassAlgorithm::canAnalyze() const {
	return hasAllRaws();
    }
    
    shared_ptr<SctData::TestResult> FullBypassAlgorithm::createTestResult() const {
	return shared_ptr<FullBypassTestResult> (new FullBypassTestResult());
    }
    
    void FullBypassAlgorithm::analyze() {
	bool debug=false;

	shared_ptr<FullBypassTestResult> result =  dynamic_pointer_cast<FullBypassTestResult> ( getTestResult() );
	result->setScanVariable(getRaw(0)->getHeader().getVariable());	
	
	if (debug) cout << "FullBypassAlgorithm" << endl;

	const unsigned nconfig=getRaw(0)->getPoints().getNPoints();
	result->setNConfig(nconfig);

	short chip_status[nconfig][nChipModule];

	// -2=nodata, -1=not_analysed, 0=fail, 1=pass
	short module_status[result->getNScans()];

	int direct_token[result->getNScans()][nChipModule];
	int bypass_token[result->getNScans()][nChipModule];

	// Algorithm by Peter Phillips & Lars Eklund.

	for (unsigned index=0; index<result->getNScans() ; ++index){
	    shared_ptr<const RawScanResult> raw=getRaw(index);

	    if (debug) cout << "Looking at scan #" << index << endl;

	    if (!raw.get()) {
		ostringstream os;
		os << "FullBypassAlgorithm: unable to get scan at index " << index <<ends;
		throw IllegalStateError(os.str(), __FILE__, __LINE__);
	    }

	    OccupancyProjector op(*raw);

	    short conf_status[nconfig][nLinkModule];

	    for (unsigned ichip=0; ichip<nChipModule; ++ichip){

		auto_ptr<TH1> hist = op.getOccupancy("hist", ModuleElement::Chip(ichip));
		
		for (unsigned iconfig=0; iconfig<nconfig; ++iconfig){
		    
		    double content = hist->GetBinContent(iconfig+1);
		    
// if (debug) cout << "content of bin "<< iconfig << " = " << content << endl;
		    
		    if (content > 0.90 && 
			content < 1.03 ) {      // CORRECT behaviour
			chip_status[iconfig][ichip]=1;
		    } else if (content <= 0.90 ) { // chip dead
			chip_status[iconfig][ichip]=0;
		    } else {
			chip_status[iconfig][ichip]=2;           // undefined status
		    } // end of if
		    
		} // loop over confs
		
	    } //loop over configs

	    //
	    // Now attempt to find the sum of 2^ichip for each link.
	    //
#ifndef NDEBUG
	    if (debug) cout << "finding 2^chip" << endl;
#endif
	    
	    for (unsigned iconfig=0; iconfig<nconfig; ++iconfig){
		int conf_sum[nLinkModule];
		for (short unsigned ilink=0; ilink<nLinkModule; ++ilink){
		    conf_sum[ilink]=0;
		}
		
		for (short unsigned ilink=0; ilink<nLinkModule; ++ilink){
		    for (short unsigned ichip=0; ichip<nChipLink; ++ichip){
			unsigned the_chip = ichip+nChipLink*ilink;
//			if(debug) cout << "chip_status" << the_chip << "="
//				       << chip_status[iconfig][the_chip]<<endl;
			if (chip_status[iconfig][the_chip] == 0){
			    // dead - do nothing
			} else if (chip_status[iconfig][the_chip] == 1){
			    conf_sum[ilink] += (1 << ichip);
			} else if (chip_status[iconfig][the_chip] == 2){
			    conf_sum[ilink] = -1; // BAD!
			    break;
			}
		    }
//		    if (debug) cout << "conf_sum[" << ilink << "]= " << conf_sum[ilink] <<endl;
		}
		
		// check matches predicted pattern.
		if (iconfig<20){ // reading-out links in parallel

		    for (short unsigned ilink=0; ilink<nLinkModule; ++ilink){
			if (conf_sum[ilink]==(int)getConf(iconfig)) { 
			    conf_status[iconfig][ilink] = 1;
			} else {
			    conf_status[iconfig][ilink]=0;
			}
		    }
		    
		} else if (iconfig<36){

		    if ( conf_sum[iconfig%2]==(int)getConf((iconfig-(20+iconfig%2) ) / 2 + 12 ) 
			 && conf_sum[(iconfig+1)%2]==2) {
			
			conf_status[iconfig][iconfig%2] = 1;   // reading out through link iconfig%2
		    } else {  
			conf_status[iconfig][iconfig%2] = 0;			
		    }
		    conf_status[iconfig][(iconfig+1)%2] = -1;  // other link no data
		    
		    
		} else if (iconfig<62 ){

		    if ( conf_sum[iconfig%2]==(int)getConf((iconfig-(36+iconfig%2) ) / 2 + 7 ) 
			 && conf_sum[(iconfig+1)%2]==1 ) {
			conf_status[iconfig][iconfig%2] = 1;   // reading out through link iconfig%2
		    } else {  
			conf_status[iconfig][iconfig%2] = 0;			
		    }
		    conf_status[iconfig][(iconfig+1)%2] = -1;  // other link no data		    
		    
		} else {
		    ostringstream os;
		    os << "FullBypassAlgorithm: iconfig out too big. max=61, val=" << iconfig <<ends;
		    throw IllegalStateError(os.str(), __FILE__, __LINE__);
		}
		
#ifndef NDEBUG
		if (debug) {
		    for (unsigned ilink=0; ilink<nLinkModule; ++ilink){
//			cout << "conf_status: iconfig=" <<iconfig
//			     << ", ilink="<<ilink<<", value = " << conf_status[iconfig][ilink] <<endl;
		    }
		}
#endif


	    } // loop over configs
	    
	    if (debug) cout << "\n\n------------------> Checking chip status" << endl;

	    int chip_accessible[nChipModule];
	    //determine if a chip is acessible in the read-out chain.
	    for (short unsigned ilink=0; ilink<nLinkModule; ++ilink){
		for (short unsigned ichip=0; ichip<nChipLink; ++ichip){
		    short unsigned the_chip=ichip+nChipLink*ilink;
		    chip_accessible[the_chip] = -1;
		    for (int iconfig=(int)sumFib(ichip); iconfig<(int)sumFib(ichip+1); ++iconfig ){
			if (conf_status[iconfig][ilink]==1){
			    chip_accessible[the_chip]=iconfig;
			}
		    }
		}
	    }
	    
	    if (debug) cout << "\n\n--------------> finding token status" << endl;
	    
	    // loop over token/data links to determine if they are working/accessible

	    for (short unsigned ilink=0; ilink<nLinkModule; ++ilink){
		for (short unsigned ichip=0; ichip<nChipLink; ++ichip){
		    short unsigned the_chip=ichip+nChipLink*ilink;
		    if (chip_accessible[the_chip] != -1){ //token link accessible?
			switch(ichip){
			case 4: {
			    direct_token[index][the_chip] =
				conf_status[ getConfNo(getConf(chip_accessible[the_chip]) + (1 << (ichip+1))) ][ilink];


			    unsigned test_config=36 + 2*(chip_accessible[the_chip] - 7) + ilink;
			    if (test_config<nconfig){
				bypass_token[index][the_chip] =
				    conf_status[test_config][ilink];
			    }else{
				bypass_token[index][the_chip] = -2;
			    }
			    break;
			}
			case 5: {
			    unsigned test_config;
			    test_config=36 + 2*(chip_accessible[the_chip] - 7) + ilink;
			    
			    if (test_config<nconfig) {
				direct_token[index][the_chip] =
				    conf_status[test_config][ilink];
			    } else {
				direct_token[index][the_chip] = -2; // not-tested
			    }

			    test_config = 20 + 2*(chip_accessible[the_chip] - 12) + ilink;
			    if (test_config<nconfig) {
				bypass_token[index][the_chip] = 
				    conf_status[test_config][ilink];				
			    } else {
				bypass_token[index][the_chip] = -2;
			    }
			    break;
			}
			default: {
			    direct_token[index][the_chip] =
				conf_status[ getConfNo(getConf(chip_accessible[the_chip]) + (int)pow(2.,(ichip+1))) ][ilink];
			    bypass_token[index][the_chip] =
				conf_status[ getConfNo(getConf(chip_accessible[the_chip]) + (int)pow(2.,(ichip+2))) ][ilink];
			    break;
			}
			}
		    } else{        // Chip not accessible, but do we really test that
			// configuration?
			if (debug) {
			    cout << "\n\n -------------> CHIP " << ichip 
				 << " not accessikble;" << endl; 
			}
			switch(ichip){
			case 4:
			    if(nconfig>=20){
				direct_token[index][the_chip] = -1;
			    }else{
				direct_token[index][the_chip] = -2;
			    }
			    if(nconfig>=(unsigned)(45+ilink)){
				bypass_token[index][the_chip] = -1;				
			    }else{
				bypass_token[index][the_chip] = -2;
			    }
			    break;
			    
			case 5:
			    if(nconfig>=(unsigned)(61+ilink)){
				direct_token[index][the_chip] = -1;
			    }else{
				direct_token[index][the_chip] = -2;
			    }
			    if(nconfig>=(unsigned)(35+ilink)){ ///@note possible bug?
				bypass_token[index][the_chip] = -1;
			    }else{
				bypass_token[index][the_chip] = -2;
			    }
			    break;
			    
			default:
			    if(sumFib(ichip+2)+ilink<=nconfig+1){
				direct_token[index][the_chip] = -1; // Not accessible
			    }else{
				direct_token[index][the_chip] = -2; // Not tested
			    }
			    if(sumFib(ichip+3)+ilink<=nconfig+1){
				bypass_token[index][the_chip] = -1;
			    }else{
				bypass_token[index][the_chip] = -2;
			    }
			    break;
			} // end of switch ichip
		    } // end of if chip_accessible
		} // loop over chips
	    } // loop over links
	
	    if (debug) cout << "\n\n-------------------------> Check if passed" << endl; 
	    // Finally , has the module passed?
	    module_status[index]=1;
	    for (unsigned ichip=0; ichip<nChipModule; ++ichip){
		if (direct_token[index][ichip] == -1 || bypass_token[index][ichip] == -1
		    || direct_token[index][ichip] == 0 || bypass_token[index][ichip] == 0 ){
		    if (debug) cout << "Problem with chip " << ichip 
				    << "\tD=" << direct_token[index][ichip] 
				    << "\tB=" << bypass_token[index][ichip] << endl;
		    module_status[index] = 0;
		}
	    }
	    if (debug) {
		cout << ( (module_status[index]==1) ? "PASS" : "FAIL" ) 
		     << "\n----------------------------------------\n" << endl;
	    }
	}// loop over vdd

	if (debug) cout << "\n\n---------------> looping over vdd" << endl;

	// Check overall status of chips - do they work at any bias?
	for (unsigned ichip=0; ichip<nChipModule; ++ichip){
	    SctData::FullBypassTestResult::ChipFBResult& cr = result->getChipResult(ichip);

	    bool keep_going_with_chip=true;
	    for (unsigned index=0; index<result->getNScans() && keep_going_with_chip; ++index){
		
		switch (direct_token[index][ichip]){              
// not tested:
		case -2 :
		    cr.vdd_direct=0; 
		    cr.status_direct=-2; 
		    keep_going_with_chip=false;
		    break;

// working at this voltage:
		case +1 : 
#ifndef NDEBUG
		    if (debug){
			cout << "Chip " << ichip << " working at vdd="; 
			cout << result->getTestPointAt(index) << " index="<<index << endl; 
		    }
#endif
		    cr.vdd_direct=result->getTestPointAt(index); 
		    cr.status_direct=1; 
		    break;
		    
// failure is true malfunction:
		case  0 : 
		    cr.status_direct=0; 
		    keep_going_with_chip=false;
		    break;

// not accessible:
		case -1 : 
		    cr.status_direct=-1; 
		    keep_going_with_chip=false;
		    break;
		default: 
		    cerr << __FILE__ << ":" << __LINE__ 
			 << " ERROR direct_token["<<index<<"]["<<ichip
			 << "] =" << direct_token[index][ichip] << endl;
		    break;
		}
	    } // loop over bias

	    // check at max voltage.
	    if (direct_token[0][ichip]==0) {
		cr.status_direct=-3;
		result->getDefects().addDefect(Defect(StandardDefects::TOKEN, ModuleElement::Chip(ichip)));
	    }
	} // loop over chips

	// NOW check the bypass token
	for (unsigned ichip=0; ichip<nChipModule; ++ichip){
	    SctData::FullBypassTestResult::ChipFBResult& cr = result->getChipResult(ichip);

	    cr.status_bypass=-3;
	    cr.vdd_bypass=10.;

	    bool keep_going_with_chip=true;
	    for (unsigned index=0; index<result->getNScans() && keep_going_with_chip; ++index){
		switch (bypass_token[index][ichip]){
		case -2 : 
		    cr.vdd_bypass=0.;
		    cr.status_bypass=-2;
		    keep_going_with_chip=false;
		    break;
		case 1:
		    cr.vdd_bypass=result->getTestPointAt(index);
		    cr.status_bypass=1;
		    break;
		case 0:
		    cr.status_bypass=0;
		    keep_going_with_chip=false;
		    break;
		case -1:
		    cr.status_bypass=-1;
		    keep_going_with_chip=false;
		    break;
		default: 
		    cerr << __FILE__ << ":" << __LINE__ 
			 << " ERROR bypass_token["<<index<<"]["<<ichip
			 << "] =" << bypass_token[index][ichip] << endl;
		    break;
		}
	    } // loop over scans
	    // check at max voltage.
	    if (bypass_token[0][ichip]==0) {
		cr.status_bypass=-3;
		result->getDefects().addDefect(Defect(StandardDefects::RTOKEN, ModuleElement::Chip(ichip)));
	    }
	} // loop over chips
	
	if (debug) cout << "\n\n---------------------> nearly done" << endl;

	// find the nominal voltage.
	short i_vdd_nom;
	for (unsigned index=0; index<result->getNScans(); ++index){
	    if ( fabs(result->getTestPointAt(index)-getVddNom()) < 0.05 ) {
		result->setVddNom(result->getTestPointAt(index)) ;
		i_vdd_nom=index;
		if (module_status[index]==1){
		    result->setPassed(true);
		}
	    }
	    if (module_status[index]!=1) result->setProblem(true);
	}
	if (debug) cout << "Module pass = " << (result->getPassed() ? "TRUE" : "FALSE") << endl;
    }
    
    unsigned FullBypassAlgorithm::getConfNo(const unsigned token){
	const unsigned MAX_CONF=60;
	for(unsigned i=0;i<MAX_CONF;i++){
	    if(getConf(i)==token){
		return i;
	    }  
	}
	ostringstream os;
	os << "No configuration matches your token="<<token; 
	throw IllegalStateError(os.str(),__FILE__,__LINE__);
    }
    
    
    unsigned FullBypassAlgorithm::getConf(const unsigned iconf) {
	const unsigned MAX_CONF=60;
	if (iconf<0 || iconf>= MAX_CONF) {
	    throw Sct::OutOfRangeError<int> ("FullBypassAlgorithm::getConf",__FILE__, __LINE__, iconf, 0, MAX_CONF-1);
	}
	const unsigned  conf[MAX_CONF]
	    = {  1,   3,   5,   7,  11,  13,  15,  21,  23,  27,
		 29,  31,  43,  45,  47,  53,  55,  59,  61,  63,
		 85,  87,  91,  93,  95, 107, 109, 111, 117, 119,
		 123, 125, 127, 171, 173, 175, 181, 183, 187, 189,
		 191, 213, 215, 219, 221, 223, 235, 237, 239, 245,
		 247, 251, 253, 255,   0,   0,   0,   0,   0,   0};
	
	return conf[iconf];
	/* look-up table used for speed. Equivalent algorithm is :
	   unsigned conf[MAX_CONF];
	   int confno = -1;
	   
	   for(int i=0;i<MAX_CONF;i++){
	   conf[i] = 0;
	   }
	   
	   for(int ichip=1;ichip<=8;ichip++){
	   if(ichip==1){
	   conf[0] = 0x1;
	   } else if(ichip==2){
	   conf[1] = 0x3;
	   } else{
	   for(int i=0;i<Fib(ichip-2);i++){
	   conf[(int)sumFib(ichip-1)+i] = conf[(int)sumFib(ichip-3)+i] + (0x1 << (ichip-1));
	   }
	   for(int i=0;i<Fib(ichip-1);i++){
	   conf[(int)sumFib(ichip-1)+(int)Fib(ichip-2)+i] = conf[(int)sumFib(ichip-2)+i] + (0x1 << (ichip-1));
	   }
	   }
	   }
	*/
    }

    unsigned FullBypassAlgorithm::sumFib(const unsigned int n) {
	// Return the sum of all Fibonacci numbers up to n
	// new version by PWP 01.11.01
	unsigned i = 1;
	unsigned m = 1;
	unsigned m1 = 1;
	unsigned m2 = 0;
	unsigned s  = 1;
	if(n<1) return 0;
	while(i<n){
	    m = m1 + m2;
	    m2 = m1;
	    m1 = m;
	    s+=m;
	    i++;
	}
	return s;
    }

} // end of namespace SctAnalysis
