/*! This is a test program to read a PrimList from an XML file and send it to the ROD.
It follows the format in Xml/README.rules and illustrated by Xml/Example.xml.
It retrieves any reply buffer generated by the PrimList and prints it to the screen.

This program can be called with optional arguments as follows:
-n :tells it not to initialize the ROD before sending it to the PrimList.
-s# where # is an integer between 1 and 21: set the VME slot number.
-f<filename>: set the name of the XML file to be read to be <filename>

If either -s or -f is missing, the user will be prompted for a value.
No space can come between -s and the slot number or between -f and the file name.

Examples:

./XmlTest -n -s7 -fXml/Exam[ple.xml
  This reads the file Xml/Example.xml and sends it to the ROD in slot 7 without calling
  the initialize() method of RodModule.
  
./XmlTest
  This executes the program, prompting the user for all information and initializing the
  ROD.
  
*/

#include <iostream>
using namespace std;

#include <ctype.h>
#include <string>

#include "RodModule.h"

#include "primParams.h"
#include "RCCVmeInterface.h"
#include "parameters.h"

int main(int argc, char *argv[]) {

using namespace SctPixelRod;

  RodPrimList primList(1);                      // Primitive List
  long myTextLength;                            // Actual length of text message
  TEXT_BUFFER_TYPE myTextType;                  // Buffer type for latest message
  PrimState returnPState;
  TextBuffState returnTState;
  std::string fileName(""), option;
  bool initRod = true;
  int slot = -1;
  unsigned long baseAddress;

  if (argc > 1) {
    for (int i=1; i<argc; i++) {
      option = argv[i];
      if (option[0] != '-') break;
      switch (option[1]) {
        case 'n': {
          initRod = false;
          break;
        }
        case 's': {
          slot = atoi(option.substr(2).c_str());
          break;
        }
        case 'f': {
          fileName = option.substr(2);
          break;
        }
        default: {
          break;
        }
      }
    }
  }
  
// Prompt for slot number
  if (slot < 0 ) {
    cout << "Enter slot number (decimal):"; 
    cin >> slot;
    while ((slot < 1) || (slot > 21)) {
      cout << "Slot number out or range [1:21], re-enter: ";
      cin >> slot;
    }
  }
  baseAddress = slot << 24;

// Create VME interface
  RCCVmeInterface *vme1;
  try {
    vme1 = new RCCVmeInterface();
  }
  catch (VmeException &v) {
    cout << "VmeException creating VME interface." << endl;
    cout << "ErrorClass = " << v.getErrorClass() << endl;
    cout << "ErrorCode = " << v.getErrorCode() << endl; 
  }
  
// Create RodModule and initialize it
  RodModule* rod0 = new RodModule(baseAddress, mapSize, *vme1, numSlaves);
  if (initRod) {
    try{
      rod0->initialize();
    }
    catch (HpiException &h) {
      cout << h;
    }
    catch (VmeException &v) {
      cout << "VmeException creating RodModule." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
    catch (RodException &r) {
    cout << r;
    }
  }

// Create a buffer for text messages
  char * myTextBuffer;
  try {
    myTextBuffer= new char[TEXT_BUFF_SIZE];
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate text buffer in main.";
  }
  
// Read an XML file into primlist buffer
  if (fileName == "") {
    cout << "Enter name of XML file: ";
    cin >> fileName;
  }
  try {
    primList.buildFromXml(fileName);
  }
  catch (PrimListException &p) {
    cout << p.getDescriptor() << " ";
    cout << p.getData1() << ", " << p.getData2() << "\n";
  };
  try {
    rod0->sendPrimList(&primList);
  }
  catch (HpiException &h) {
    cout << h;
  }
  catch (VmeException &v) {
    cout << "VmeException in SendPrimList." << endl;
    cout << "ErrorClass = " << v.getErrorClass() << endl;
    cout << "ErrorCode = " << v.getErrorCode() << endl; 
  };
  
// Wait for ROD to begin executing and then wait for it to finish executing
// Check for error messages in text buffer and read them out if they exist.
// Note: this is NOT how primHandler and textHandler will be used once we
// go to threads.  This is for debugging the code only.
  do {
    try {
      returnPState = rod0->primHandler();
    }
    catch (VmeException &v) {
      cout << "VmeException in first primHandler call." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
    try {
      returnTState = rod0->textHandler();
    }
    catch (VmeException &v) {
      cout << "VmeException in first textHandler call." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
    if (returnTState == TEXT_RQ_SET) {
      do {
        try {
          returnTState = rod0->textHandler();
          }
        catch (VmeException &v) {
          cout << "VmeException in second textHandler call." << endl;
          cout << "ErrorClass = " << v.getErrorClass() << endl;
          cout << "ErrorCode = " << v.getErrorCode() << endl; 
        }
      } while (returnTState != TEXT_READOUT);
      rod0->getTextBuffer(myTextBuffer, myTextLength, myTextType);
      rod0->clearTextBuffer();
      for (int i=0; i<myTextLength; i++) {
        cout << myTextBuffer[i];
        if (0==(i+1)%64) cout << endl;
        }
      cout << endl; 
    }
  } while (returnPState != PRIM_EXECUTING); 
  do {
    try {
      returnPState = rod0->primHandler();
    }
    catch (RodException &r) {
    cout << r;
    }
    catch (VmeException &v) {
      cout << "VmeException in second primHandler call." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
  } while (returnPState != PRIM_WAITING); 
  
// Retrieve output buffer
  RodOutList* outList = rod0->getOutList();
  
// Print results (User processing of outList)
  UINT32 outLength = UINT32(outList->getLength());
  unsigned long* outBody = outList->getBody();
  UINT32 outIndex = UINT32(outBody[1]);
  UINT32 outNumPrims = outBody[2];
  UINT32 outPrimVersion = outBody[3];
  cout << "outLength = " << outLength << ", outIndex = " << outIndex << 
          ", outNumPrims = " << outNumPrims << ", outPrimVersion = " << 
          outPrimVersion <<'\n';
  int outPtr = 4;
  for (UINT32 j=0; j<outNumPrims; j++) {
    UINT32 primLength = outBody[outPtr++];
    UINT32 primIndex = outBody[outPtr++];
    UINT32 primId = outBody[outPtr++];
    UINT32 primVersion = outBody[outPtr++];
    cout << "primLength = " << primLength << ", primIndex = " << primIndex << 
            ", primId = " << primId << ", primVersion = " << primVersion << '\n';
    cout << "ECHO data: " << endl;
    hex(cout);
    for (UINT32 i=0; i<primLength-4; i++) {
      cout.width(8);
      cout << outBody[i+outPtr] <<" ";
      if (0 == (i+1)%8) cout << endl; 
    };
    if (0 != (primLength-4)%8) cout << endl;
    dec(cout);
    outPtr += primLength-4;
  };
  
// Clean up: clear primList, delete the outList
  primList.clear();
  delete [] myTextBuffer;
  rod0->deleteOutList();
  
// Delete the ROD and VME objects before exiting
  delete rod0;
  delete vme1;

  return 0;  
}


